/*
* JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
*
*    @author    Jpresta
*    @copyright Jpresta
*    @license   See the license of this module in file LICENSE.txt, thank you.
*/

const MecaFaces = {
    WAITING: 'meca-waiting',
    HAPPY: 'meca-happy',
    SHOWING: 'meca-showing',
    SAD: 'meca-sad',
}

const DocFaces = {
    WAITING: 'doc-waiting',
    HAPPY: 'doc-happy',
    SAD: 'doc-sad',
    CRYING: 'doc-crying',
}

const ProblemStates = {
    TO_FIX: 0,
    CANNOT_FIX: 1,
    FIXED: 2,
    ERROR_IN_FIX: 3,
}

const PageStates = {
    IDLE: 'idle',
    WAITING_SETTINGS_REVIEW: 'settings',
    MODIFICATION_RUNNING: 'modifying',
    DIAGNOSTIC_RUNNING: 'diagnostic_running',
    DIAGNOSTIC_RUNNING_SINGLE: 'diagnostic_running_single',
    DIAGNOSTIC_STOPPING: 'diagnostic_stopping',
    REPAIR_RUNNING: 'repair_running',
    REPAIR_RUNNING_SINGLE: 'repair_running_single',
    REPAIR_STOPPING: 'repair_stopping',
};

const allowedTransitions = {
    [PageStates.IDLE]: [PageStates.IDLE, PageStates.WAITING_SETTINGS_REVIEW, PageStates.DIAGNOSTIC_RUNNING, PageStates.DIAGNOSTIC_RUNNING_SINGLE, PageStates.REPAIR_RUNNING, PageStates.REPAIR_RUNNING_SINGLE, PageStates.MODIFICATION_RUNNING],
    [PageStates.WAITING_SETTINGS_REVIEW]: [PageStates.IDLE],
    [PageStates.DIAGNOSTIC_RUNNING]: [PageStates.DIAGNOSTIC_STOPPING, PageStates.IDLE],
    [PageStates.DIAGNOSTIC_STOPPING]: [PageStates.IDLE],
    [PageStates.REPAIR_RUNNING]: [PageStates.REPAIR_STOPPING, PageStates.IDLE],
    [PageStates.REPAIR_STOPPING]: [PageStates.IDLE],
    [PageStates.MODIFICATION_RUNNING]: [PageStates.IDLE],
    [PageStates.DIAGNOSTIC_RUNNING_SINGLE]: [PageStates.DIAGNOSTIC_STOPPING, PageStates.IDLE],
    [PageStates.REPAIR_RUNNING_SINGLE]: [PageStates.REPAIR_STOPPING, PageStates.IDLE],
};

let currentState = PageStates.IDLE;

let datasControlsTable = null;
let datasProblemsTable = null;
let progressCount = 0;
let progressTotalCount = 0;

function setState(newState) {
    if (!allowedTransitions[currentState].includes(newState)) {
        console.warn(`Invalid transition from state '${currentState}' to state '${newState}'`);
        return false;
    }

    currentState = newState;

    // Reset all UI flags/classes
    $('#content')
        .removeClass('diagnostic_running diagnostic_stopping repair_running repair_stopping');

    // Apply new state
    $('#content').addClass(newState);
    switch (newState) {
        case PageStates.IDLE:
            resetProgress();
            break;
        case PageStates.DIAGNOSTIC_RUNNING:
            progressTotalCount = 0;
            progressCount = 0;
            break;
        case PageStates.DIAGNOSTIC_STOPPING:
            break;
        case PageStates.REPAIR_RUNNING:
            progressTotalCount = 0;
            progressCount = 0;
            break;
        case PageStates.REPAIR_STOPPING:
            break;
        case PageStates.MODIFICATION_RUNNING:
            break;
        case PageStates.DIAGNOSTIC_RUNNING_SINGLE:
            progressTotalCount = 1;
            progressCount = 0;
            break;
        case PageStates.REPAIR_RUNNING_SINGLE:
            progressTotalCount = 1;
            progressCount = 0;
            break;
    }

    // Update buttons availability
    updateUIByState();

    return true;
}

function updateUIByState() {
    const hasProblems = problem_count > 0;

    // Stats (problem count)
    updateStats();

    $('#startDiagnostic').prop('disabled', currentState !== PageStates.IDLE || control_count === 0);
    $('#stopDiagnostic').prop('disabled', currentState !== PageStates.DIAGNOSTIC_RUNNING && currentState !== PageStates.DIAGNOSTIC_RUNNING_SINGLE);
    $('#startRepair').prop('disabled', currentState !== PageStates.IDLE || problem_counts[ProblemStates.TO_FIX] === 0);
    $('#stopRepair').prop('disabled', currentState !== PageStates.REPAIR_RUNNING && currentState !== PageStates.REPAIR_RUNNING_SINGLE);
    switch (currentState) {
        case PageStates.IDLE:
        case PageStates.WAITING_SETTINGS_REVIEW:
            $('#diagnostic-progress').hide();
            $('#startDiagnostic').show();
            $('#stopDiagnostic').hide();
            $('#stoppingDiagnostic').hide();
            $('#startRepair').show();
            $('#stopRepair').hide();
            $('#stoppingRepair').hide();
            break;
        case PageStates.DIAGNOSTIC_RUNNING:
            $('#diagnostic-progress').show();
            $('#startDiagnostic').hide();
            $('#stopDiagnostic').show();
            break;
        case PageStates.DIAGNOSTIC_STOPPING:
            $('#stopDiagnostic').hide();
            $('#stoppingDiagnostic').show();
            break;
        case PageStates.REPAIR_RUNNING:
        case PageStates.REPAIR_RUNNING_SINGLE:
            $('#diagnostic-progress').show();
            $('#startRepair').hide();
            $('#stopRepair').show();
            break;
        case PageStates.REPAIR_STOPPING:
            $('#stopRepair').hide();
            $('#stoppingRepair').show();
            break;
        case PageStates.DIAGNOSTIC_RUNNING_SINGLE:
            $('#diagnostic-progress').show();
            $('#startDiagnostic').hide();
            $('#stopDiagnostic').show();
            break;
    }
}

// ---------------------------
// Initialization
// ---------------------------
$(document).ready(function () {
    // Trigger all the logic behind states
    setState(needs_settings_review ? PageStates.WAITING_SETTINGS_REVIEW : PageStates.IDLE)

    // Move FAQ button into toolbar
    $('#btn-doctor-faq-li').prependTo('.btn-toolbar ul.nav');

    // Init DataTables
    $.fn.dataTable.ext.errMode = 'none';
    datasControlsTable = initControlsTable();

    // Bind UI events
    bindControlsEvents();
});

// ---------------------------
// Controls DataTable
// ---------------------------
function initControlsTable() {
    return $('#datasControlsTable')
        .on('error.dt', handleDataTableError)
        .DataTable({
            processing: true,
            serverSide: true,
            searching: true,
            ajax: doctor_datas_controls_url,
            columns: [
                { width: '2rem', className: 'tdid' },
                { orderable: false, render: function (data, type, row) { return `<span title="${row[8]}">${row[1]}</span>`;} },
                { width: '7rem', className: 'tddate', render: function (data) { return data ? data : '-'; }  },
                { width: '7rem', className: 'tdcenter', render: function (data, type, row) { return row[2] ? data : '-'; } },
                { width: '7rem', className: 'tdcenter', render: function (data, type, row) { return row[2] ? data : '-'; } },
                { width: '7rem', className: 'tdcenter', render: function (data, type, row) { return row[2] ? data : '-'; } },
                { width: '7rem', className: 'tdcenter', render: function (data, type, row) { return row[2] ? data : '-'; } },
                {
                    data: null,
                    width: '7rem',
                    className: 'tdright js-actions',
                    orderable: false,
                    render: function (data, type, row) {
                        return `
                <button title="${lbl_run_diagnostic}"
                        class="btn btn-xs btn-primary js-btn-evaluate"
                        data-id-control="${row[0]}" data-control-name="${row[1]}">
                  <i class="icon-play"></i>
                </button>
                <button title="${lbl_delete_ctrl}"
                        class="btn btn-xs btn-danger js-btn-delete-ctrl"
                        data-id-control="${row[0]}">
                  <i class="icon-trash"></i>
                </button>`;
                    }
                }
            ],
            order: [[4, 'desc']],
            language: getLocalizationSettings(),
            dom: 'Blfrtip',
            lengthMenu: getLengthMenuOptions(),
            buttons: [],
        });
}

// ---------------------------
// Problems DataTable
// ---------------------------
function initProblemsTable(id_control) {
    return $('#datasProblemsTable')
        .on('error.dt', handleDataTableError)
        .DataTable({
            processing: true,
            serverSide: true,
            searching: true,
            ajax: doctor_datas_problems_url + '&id_control=' + id_control,
            columns: [
                { width: '7rem', className: 'tdid'},
                { orderable: false,
                    render: function (data, type, row) {
                        // Parse JSON from column 4 : problem_messages_json
                        let messages = {};
                        try {
                            messages = JSON.parse(row[4] || '{}');
                        } catch (e) {
                            // Invalid JSON → show only main value
                            return data || '';
                        }

                        // Icons for each message type
                        const icons = {
                            success: '✅',
                            info: 'ℹ️',
                            warning: '⚠️',
                            error: '❌'
                        };

                        // Background colors for each message type
                        const bgColors = {
                            success: '#d4edda', // light green
                            info: '#d1ecf1',    // light blue
                            warning: '#fff3cd', // light yellow
                            error: '#f8d7da'    // light red
                        };

                        // Keep the HTML from `data` (may contain entities or tags)
                        let html = data || '';

                        // Build the message blocks
                        let msgHtml = '';
                        for (const type in messages) {
                            const arr = messages[type];
                            if (Array.isArray(arr) && arr.length > 0) {
                                for (const msg of arr) {
                                    // Escape only message text (not icons)
                                    const safeMsg = $('<div>').text(msg).html();
                                    msgHtml += `
                        <div style="
                            background:${bgColors[type] || '#eee'};
                            border-radius:4px;
                            padding:3px 6px;
                            margin:2px 0;
                            font-size:0.9em;
                            line-height:1.3em;
                        ">
                            ${icons[type] || ''} ${safeMsg}
                        </div>`;
                                }
                            }
                        }

                        // If no messages, return only main value
                        if (!msgHtml) return html;

                        // Combine main content and formatted messages
                        return `
                            ${html}
                            <div class="msg-list" style="margin-top:6px;">
                                ${msgHtml}
                            </div>
                        `;
                    }
                },
                {
                    width: '7rem',
                    className: 'tdcenter',
                    render: function (data) {
                        switch (String(data)) {
                            case '0': return "👍 " + lbl_fixable;
                            case '1': return "⚠️ " + lbl_not_fixable;
                            case '2': return "✅ " + lbl_fixed;
                            case '3': return "❌ " + lbl_fix_error;
                            default:  return "?";
                        }
                    }
                },
                {
                    data: null,
                    width: '7rem',
                    className: 'tdright js-actions',
                    render: function (data, type, row) {
                        let out = '';
                        if (String(row[2]) === '0') {
                            out += `<button title="${lbl_fix_pb}"
                                class="btn btn-xs btn-success js-btn-repair"
                                data-id-problem="${row[0]}"><i class="icon-wrench"></i></button>`;
                        }
                        out += `<button title="${lbl_delete_pb}" class="btn btn-xs btn-danger js-btn-delete-pb"
                              data-id-problem="${row[0]}"><i class="icon-trash"></i></button>`;
                        return out;
                    }
                }
            ],
            order: [[0, 'desc']],
            language: getLocalizationSettings(),
            dom: 'Blrtip',
            lengthMenu: getLengthMenuOptions(),
            buttons: [],
            initComplete: function () {
                const api = this.api();
                const colIndex = 2;
                const values = [
                    {value: ProblemStates.FIXED, text: lbl_fixed},
                    {value: ProblemStates.TO_FIX, text: lbl_fixable},
                    {value: ProblemStates.CANNOT_FIX, text: lbl_not_fixable},
                    {value: ProblemStates.ERROR_IN_FIX, text: lbl_fix_error}
                ];

                const $select = $('<select><option value="">-</option></select>');

                // Ajoute les options depuis ton tableau d'objets
                $.each(values, function (_, item) {
                    $select.append(
                        $('<option>', {
                            value: item.value,
                            text: item.text
                        })
                    );
                });

                // Stop propagation to prevent DataTables redraw on click
                $select.on('click mousedown', function (e) {
                    e.stopPropagation();
                });

                $select
                    .appendTo($(api.column(colIndex).header()))
                    .on('change', function () {
                        const val = $.fn.dataTable.util.escapeRegex($(this).val());
                        api.column(colIndex)
                            .search(val || '', false, false)
                            .draw();
                    });
            }
        });
}

// ---------------------------
// Event Bindings (start/stop, delete, repair, row expand)
// ---------------------------
function bindControlsEvents() {
    $('#startDiagnostic').on('click', function (e) {
        e.preventDefault();
        startDiagnostic();
    });
    $('#stopDiagnostic').on('click', function (e) {
        e.preventDefault();
        stoppingDiagnostic();
    });
    $('#openRepairInfo').on('click', function (e) {
        e.preventDefault();
        backupCallback = null;
        $('#backupCancelBtn').hide();
        $('#backupConfirmBtn').hide();
        $('#backupCloseBtn').show();
        $('#backupWarningModal').modal({
            backdrop: 'static',
            keyboard: false
        });
    })
    $('#startRepair').on('click', function (e) {
        e.preventDefault();
        startRepair();
    })
    $('#stopRepair').on('click', function (e) {
        e.preventDefault();
        stoppingRepair();
    })

    $('#datasControlsTable').on('click', '.js-btn-evaluate', function (e) {
        e.preventDefault();
        if (!setState(PageStates.DIAGNOSTIC_RUNNING_SINGLE)) {
            return;
        }
        diagnostic($(this).data('id-control'), $(this).data('control-name'));
    })

    $('#datasControlsTable').on('click', '.js-btn-delete-ctrl', function (e) {
        e.preventDefault();
        deleteControl($(this).data('id-control'));
    })

    $('#datasControlsTable').on('click', '.js-btn-repair', function (e) {
        e.preventDefault();
        if (!setState(PageStates.REPAIR_RUNNING_SINGLE)) {
            return;
        }
        repair($(this).data('id-problem'), $(this).closest('tr').find('td').eq(1).text().trim(), null);
    })

    $('#datasControlsTable').on('click', '.js-btn-repair-all[data-id-control]', function (e) {
        e.preventDefault();
        startRepair($(this).data('id-control'));
    })

    $('#datasControlsTable').on('click', '.js-btn-delete-pb', function (e) {
        e.preventDefault();
        deleteProblem($(this).data('id-problem'));
    });

    // Confirm → run action + save timestamp
    $('#backupConfirmBtn').on('click', function () {
        localStorage.setItem(storageKey, Date.now().toString());
        $('#backupWarningModal').modal('hide');
        if (typeof backupCallback === "function") {
            backupCallback();
            backupCallback = null;
        }
    });

    // Cancel → discard action, do not save timestamp
    $('#backupCancelBtn').on('click', function () {
        backupCallback = null;
        setState(PageStates.IDLE);
        $('#backupWarningModal').modal('hide');
        logInReport('❌ ' + lbl_repair_stopped);
    });

    $('#backupCloseBtn').on('click', function () {
        $('#backupWarningModal').modal('hide');
    });

    // Expand row to show problems
    $('#datasControlsTable > tbody').on('click', '> tr.even > td:not(.js-actions), > tr.odd > td:not(.js-actions)', function () {
        let $tr = $(this).closest('tr');
        let row = datasControlsTable.row($tr);
        let open = row.child.isShown();
        datasControlsTable.rows().every(function () {
            if (this.child.isShown()) {
                this.child.hide();
                if (datasProblemsTable) datasProblemsTable.destroy(true);
                $(this.node()).removeClass('shown');
            }
        });
        if (!open) {
            let datas = row.data();
            if (datas) {
                row.child(getTemplate('datasProblemsTableTpl')).show();
                let $child = $tr.next('tr');
                if (datas[8]) {
                    $child.find('.control_desc').first().html(datas[8]).parent().show();
                }
                if (datas[9]) {
                    $child.find('.control_howto').first().html(datas[9]).parent().show();
                }
                $child.find('.js-btn-repair-all').first().attr('data-id-control', datas[0]);
                datasProblemsTable = initProblemsTable(datas[0]);
                $tr.addClass('shown');
                $tr.next('tr').find('td').addClass('problems');
            }
        }
    });
}

// ---------------------------
// Diagnostic workflow
// ---------------------------
function diagnostic(id_control, control_name, isAfterTimedout = false) {
    if (!isAfterTimedout) {
        logInReport(` • ${lbl_running_control} "${control_name}" (${progressCount}/${progressTotalCount})`, false);
        incrementProgress(control_name, 0);
    }

    const startTime = Date.now();
    $.ajax({
        url: doctor_ajax_url,
        type: 'POST',
        dataType: 'json',
        data: {
            action: 'evaluate',
            id_control: id_control
        },
        success: function (data) {
            let timedout = data.next && Number(data.next.id_control) === Number(id_control);
            if (!timedout) {
                incrementProgress(control_name);
            }

            if (data.done) {
                if (!timedout) {
                    let msg = '';
                    if (data.done.problem_count < 2) {
                        msg = sprintf(lbl_problem_found_sing, data.done.problem_count, data.done.duration_ms);
                    } else {
                        msg = sprintf(lbl_problem_found, data.done.problem_count, data.done.duration_ms);
                    }
                    logInReport(` ⟶ ${msg}`);
                }
                // Iterate through each type of message
                Object.keys(data.done.messages).forEach(type => {
                    data.done.messages[type].forEach(msg => {
                        console.log(`[${type.toUpperCase()}] ${msg}`);
                        logInReport(`[${type.toUpperCase()}] ${msg}`);
                    });
                });
            }
            else {
                logInReport('❌ ' + lbl_report_empty);
            }

            if (data.problem_counts) {
                // Update stats
                problem_counts = data.problem_counts;
                last_full_check = data.last_full_check;
                displayStats();
            }
            if (data.next && (currentState === PageStates.DIAGNOSTIC_RUNNING
                || (currentState === PageStates.DIAGNOSTIC_RUNNING_SINGLE && timedout))) {
                // Launch next control or the same that is not finished
                diagnostic(data.next.id_control, decodeHtmlEntities(data.next.name), timedout);
            } else {
                // Finished
                logInReport('✅ ' + lbl_diag_complete);
                stopDiagnostic();
            }
        },
        error: function (xhr, status, error) {
            handleAjaxError(xhr, status, error, startTime);
            logInReport( '❌ ' + lbl_diag_stopped);
            stopDiagnostic();
        }
    });
}

function startDiagnostic() {
    if (!setState(PageStates.DIAGNOSTIC_RUNNING)) {
        logInReport('❌ ' + lbl_other_running);
        return;
    }
    logInReport('⏵ ' + lbl_diag_start);
    const startTime = Date.now();
    $.ajax({
        url: doctor_ajax_url,
        type: 'POST',
        dataType: 'json',
        data: {
            action: 'evaluate'
        },
        success: function (data) {
            progressTotalCount = data.total
            if (data.next && currentState === PageStates.DIAGNOSTIC_RUNNING) {
                diagnostic(data.next.id_control, decodeHtmlEntities(data.next.name));
            }
            else {
                logInReport('❌ ' + lbl_diag_no_control);
                stopDiagnostic();
            }
        },
        error: function (xhr, status, error) {
            handleAjaxError(xhr, status, error, startTime);
            logInReport( '❌ ' + lbl_diag_stopped);
            stopDiagnostic();
        }
    });
}
function stoppingDiagnostic() {
    setState(PageStates.DIAGNOSTIC_STOPPING);
}
function stopDiagnostic() {
    if (!setState(PageStates.IDLE)) {
        return;
    }
    if (datasControlsTable) datasControlsTable.ajax.reload();
}

function deleteProblem(id_problem) {
    if (!setState(PageStates.MODIFICATION_RUNNING)) {
        return;
    }
    $.ajax({
        url: doctor_ajax_url,
        type: 'POST',
        dataType: 'json',
        data: {
            action: 'delete_problem',
            id_problem: id_problem
        },
        success: function () {
            datasProblemsTable.ajax.reload();
        },
        complete: function () {
            setState(PageStates.IDLE);
        }
    })
}
function deleteControl(id_control) {
    if (!setState(PageStates.MODIFICATION_RUNNING)) {
        return;
    }
    $.ajax({
        url: doctor_ajax_url,
        type: 'POST',
        dataType: 'json',
        data: {
            action: 'delete_control',
            id_control: id_control
        },
        success: function () {
            datasControlsTable.ajax.reload();
        },
        complete: function () {
            setState(PageStates.IDLE);
        }
    })
}
function repair(id_problem, problem_name, id_control = null, isAfterTimedout = false) {
    handleRepairAction(function () {
        if (!isAfterTimedout) {
            logInReport(` • ${lbl_repairing_problem} "${problem_name}" (${progressCount}/${progressTotalCount})...`, true);
            incrementProgress(problem_name, 0);
        }

        const startTime = Date.now();
        $.ajax({
            url: doctor_ajax_url,
            type: 'POST',
            dataType: 'json',
            data: {
                action: 'repair',
                id_problem: id_problem,
                id_control: id_control
            },
            success: function (data) {
                let timedout = data.next && Number(data.next.id_problem) === Number(id_problem);
                if (!timedout) {
                    incrementProgress(problem_name);
                }

                if (data.done) {
                    switch (data.done.state) {
                        case ProblemStates.TO_FIX:
                            break;
                        case ProblemStates.CANNOT_FIX:
                            break;
                        case ProblemStates.FIXED:
                            break;
                        case ProblemStates.ERROR_IN_FIX:
                            break;
                    }
                    // Iterate through each type of message
                    Object.keys(data.done.messages).forEach(type => {
                        data.done.messages[type].forEach(msg => {
                            console.log(`[${type.toUpperCase()}] ${msg}`);
                            logInReport(`[${type.toUpperCase()}] ${msg}`);
                        });
                    });
                } else {
                    logInReport('❌ ' + lbl_report_empty);
                }

                if (data.problem_counts) {
                    // Update stats
                    problem_counts = data.problem_counts;
                    last_full_check = data.last_full_check;
                    displayStats();
                }
                if (data.next && (currentState === PageStates.REPAIR_RUNNING
                    || (currentState === PageStates.REPAIR_RUNNING_SINGLE && timedout))) {
                    // Launch next repair or the same that is not finished
                    repair(data.next.id_problem, decodeHtmlEntities(data.next.name), data.next.id_control, timedout);
                } else {
                    // Finished
                    logInReport('✅ ' + lbl_repair_complete);
                    stopRepair(id_control);
                }
            },
            error: function (xhr, status, error) {
                handleAjaxError(xhr, status, error, startTime);
                logInReport('❌ ' + lbl_repair_stopped);
                stopRepair(id_control);
            }
        });
    });
}
function stopRepair($id_control = null) {
    if (!setState(PageStates.IDLE)) {
        return;
    }
    if (datasControlsTable) datasControlsTable.ajax.reload();
}
function startRepair($id_control = null) {
    if (!setState(PageStates.REPAIR_RUNNING)) {
        logInReport('❌ ' + lbl_other_running);
        return;
    }

    logInReport('⏵ ' + lbl_repair_start);

    const startTime = Date.now();
    $.ajax({
        url: doctor_ajax_url,
        type: 'POST',
        dataType: 'json',
        data: {
            action: 'repair',
            id_control: $id_control
        },
        success: function (data) {
            progressTotalCount = data.total
            if (data.next && currentState === PageStates.REPAIR_RUNNING) {
                repair(data.next.id_problem, decodeHtmlEntities(data.next.name), data.next.id_control);
            }
            else {
                logInReport('❌ ' + lbl_repair_no_problem);
                stopRepair($id_control);
            }
        },
        error: function (xhr, status, error) {
            handleAjaxError(xhr, status, error, startTime);
            logInReport('❌ ' + lbl_repair_stopped);
            stopRepair($id_control);
        }
    });
}

/**
 * Handle AJAX error responses in a unified way.
 *
 * @param {object} xhr - The XMLHttpRequest object from jQuery.
 * @param {string} status - The AJAX status text (e.g., "error", "timeout").
 * @param {string} error - The error message (if any).
 * @param {number} startTime - The timestamp when the AJAX request started.
 */
function handleAjaxError(xhr, status, error, startTime) {
    // Calculate elapsed time in seconds
    const elapsed = Math.round((Date.now() - startTime) / 1000);
    logInReport(`❌ Ajax error after ${elapsed}s`);

    // Ensure xhr is an object before accessing properties
    const hasJson = typeof xhr === 'object' && xhr !== null && typeof xhr.responseJSON === 'object';
    const json = hasJson ? xhr.responseJSON : null;

    // Case 1: Server returned a JSON response containing error information
    if (json && json.error) {
        let msg = stripHtmlAndWhitespace(json.error);
        logInReport(`❌ Server error: ${msg}`);

        // Display stack trace in console if available
        if (json.error_trace) {
            console.error('Doctor error: ' + msg, json.error_trace);
        }
    }
    // Case 2: Fallback — display generic HTTP error information
    else {
        const httpCode = typeof xhr?.status === 'number' ? xhr.status : 'N/A';
        const httpText = getHttpStatusText(httpCode);
        logInReport(`❌ HTTP ${httpCode} – ${httpText}`);
        logInReport(`❌ Detail: ${status ?? 'unknown'} - ${error ?? 'no message'}`);
    }
}

/**
 * Remove HTML tags and normalize whitespace (tabs, newlines, multiple spaces).
 *
 * @param {string} html - The input string possibly containing HTML and extra spaces.
 * @returns {string} Clean plain text with normalized single spaces.
 */
function stripHtmlAndWhitespace(html) {
    if (typeof html !== 'string') return '';
    const tmp = document.createElement('div');
    tmp.innerHTML = html;
    let text = tmp.textContent || tmp.innerText || '';
    // Replace tabs, newlines, and multiple spaces with a single space
    return text.replace(/\s+/g, ' ').trim();
}

/**
 * Return a human-readable meaning for common HTTP status codes.
 */
function getHttpStatusText(code) {
    const map = {
        // Client errors
        0: "Request not sent or blocked (CORS / Network error)",
        400: "Bad Request – malformed syntax or invalid parameters",
        401: "Unauthorized – authentication required",
        403: "Forbidden – server refused the request",
        404: "Not Found – resource not found",
        408: "Request Timeout – the server took too long to respond",
        429: "Too Many Requests – rate limit exceeded",

        // Server errors
        500: "Internal Server Error – the server encountered an unexpected condition",
        501: "Not Implemented – the server does not support this method",
        502: "Bad Gateway – invalid response from an upstream server or proxy",
        503: "Service Unavailable – server temporarily overloaded or under maintenance",
        504: "Gateway Timeout – upstream server failed to respond in time",
        505: "HTTP Version Not Supported",
        507: "Insufficient Storage – the server has insufficient storage to complete the request",
        508: "Loop Detected – server detected an infinite loop while processing the request",
        510: "Not Extended – further extensions to the request are required",
        511: "Network Authentication Required – client must authenticate to gain network access",

        // Reverse proxy / CDN / WAF specific but now commonly used
        520: "Unknown Error – server returned an unexpected response",
        521: "Web Server Down – origin server refused connection",
        522: "Connection Timed Out – the server took too long to respond",
        523: "Origin Unreachable – network or DNS issue",
        524: "A Timeout Occurred – the server did not finish processing in time",
        525: "SSL Handshake Failed – secure connection could not be established",
        526: "Invalid SSL Certificate – certificate validation failed"
    };
    return map[code] || "Unknown HTTP status code";
}

function stoppingRepair() {
    setState(PageStates.REPAIR_STOPPING);
}
function updateStats() {
    $.ajax({
        url: doctor_ajax_url,
        type: 'POST',
        dataType: 'json',
        data: {
            action: 'get_stats'
        },
        success: function (data) {
            // problem_count, problem_counts and control_count
            // are defined by Prestashop in HTML document with addJsDef()
            problem_count = data.problem_count;
            problem_counts = data.problem_counts;
            control_count = data.control_count;
            last_full_check = data.last_full_check;
            displayStats();
        }
    });
}

function resetControls() {
    if (!setState(PageStates.IDLE)) {
        logInReport('❌ ' + lbl_other_running);
        return;
    }

    if (confirm(lbl_confirm_reset)) {

        // --- Block the page ---
        const overlay = $('<div>', {
            css: {
                position: 'fixed',
                top: 0,
                left: 0,
                width: '100%',
                height: '100%',
                backgroundColor: 'rgba(0,0,0,0.3)',
                zIndex: 9999,
                cursor: 'wait'
            }
        }).appendTo('body');

        const startTime = Date.now();
        $.ajax({
            url: doctor_ajax_url,
            type: 'POST',
            dataType: 'json',
            data: { action: 'reset_controls' },
            success: function (data) {
                logInReport('✅ ' + lbl_confirm_reset_ok);
            },
            error: function (xhr, status, error) {
                handleAjaxError(xhr, status, error, startTime);
            },
            complete: function () {
                location.reload();
            }
        });
    }
}

// ---------------------------
// UI helpers
// ---------------------------
function incrementProgress(label, increment = 1) {
    progressCount = progressCount + increment;
    if (progressCount > progressTotalCount) {
        progressCount = progressTotalCount;
    }
    if (progressCount < 0) {
        progressCount = 0;
    }
    $('.progress-bar')
        .css('width', Math.round((progressCount * 100) / progressTotalCount) + '%')
        .text(label + ' (' + Math.round((progressCount * 100) / progressTotalCount) + '%)');
    if (progressCount >= progressTotalCount) {
        $('.progress-bar').removeClass('progress-bar-striped progress-bar-animated')
    }
}
function resetProgress() {
    progressCount = 0;
    progressTotalCount = 0;
    $('.progress-bar')
        .addClass('progress-bar-striped progress-bar-animated')
        .css('width', '2px')
        .text('0%');
}

function logInReport(log, newline = true) {
    let report = $('#report');
    report.append(log + (newline ? '<br>' : ''));
    report.scrollTop(report[0].scrollHeight);
}

function getTemplate(tplId) {
    const template = document.querySelector('#' + tplId);
    return template.content.cloneNode(true);
}
function handleDataTableError(e, settings, techNote, message) {
    console.error('Doctor - Cannot display data: ', message);
}
function getLengthMenuOptions() { return [[10, 25, 50, 100], ['10', '25', '50', '100']]; }

function displayStats() {
    // problem_count, problem_counts, control_count and last_full_check
    // are defined by Prestashop in HTML document with addJsDef()
    // They are updated in updateStats() function.
    problem_count = 0;
    problem_counts.forEach(function (count, index) {
        $('#counter-' + index).attr('data-counter', count);
        if (index !== ProblemStates.FIXED) {
            problem_count += count;
        }
    });
    $('#ctrlCount').html(control_count);
    if (last_full_check != null && control_count > 0) {
        if (problem_count > 0) {
            $('#pbCount').html(problem_count);
        }
        else {
            $('#pbCount').html('<i class="icon-check"></i>');
        }
        if (problem_count === 0) {
            showDocFace(DocFaces.HAPPY);
            showMecaFace(MecaFaces.WAITING);
        } else if (problem_count < 100) showDocFace(DocFaces.SAD);
        else showDocFace(DocFaces.CRYING);
    } else {
        $('#pbCount').html('');
        showDocFace(DocFaces.WAITING);
    }
    if (problem_counts[ProblemStates.ERROR_IN_FIX] > 0) {
        showMecaFace(MecaFaces.SAD);
    } else if (problem_counts[ProblemStates.FIXED] > 0 && problem_counts[ProblemStates.TO_FIX] === 0) {
        showMecaFace(MecaFaces.HAPPY);
    } else if (problem_counts[ProblemStates.TO_FIX] > 0) {
        showMecaFace(MecaFaces.SHOWING);
    }
}

function showDocFace(face) {
    $('.doc-face').hide();
    $('#' + face).show();
    switch (face) {
        case DocFaces.WAITING:
            $('#doc-face-waiting').hide();
            break;
        default:
            $('#doc-face-showing').show();
            break;
    }
}

function showMecaFace(face) {
    $('.meca-face').hide();
    $('#' + face).show();
}

/**
 * Centralized function to handle repair actions
 */
let backupCallback = null;
const storageKey = "JPresta/Doctor/LastBackupWarning";

function handleRepairAction(callback) {

    // 1. If repairs are forbidden → show blocking modal
    if (!can_repair) {
        $('#cannotRepairModal').modal({
            backdrop: 'static',
            keyboard: false
        });
        setState(PageStates.IDLE);
        logInReport('❌ ' + lbl_repair_stopped);
        return;
    }

    // 2. If backup warning is needed → show modal and delay action
    const now = Date.now();
    const lastShown = localStorage.getItem(storageKey);
    if (!lastShown || (now - parseInt(lastShown, 10)) > (60 * 60 * 1000)) {
        backupCallback = callback;
        $('#backupCancelBtn').show();
        $('#backupConfirmBtn').show();
        $('#backupCloseBtn').hide();
        $('#backupWarningModal').modal({
            backdrop: 'static',
            keyboard: false
        });
    } else {
        // Already confirmed recently
        callback();
    }
}

// ---------------------------
// Utils
// ---------------------------
function sprintf(template, ...values) {
    let i = 0;
    return template.replace(/%[dsf]/g, match => {
        const val = values[i++];
        switch (match) {
            case '%d': return Number(val);
            case '%s': return String(val);
            case '%f': return parseFloat(val);
            default:   return match;
        }
    });
}

function decodeHtmlEntities(str) {
    let txt = document.createElement("textarea");
    txt.innerHTML = str;
    return txt.value;
}
