/*
* JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
*
*    @author    Jpresta
*    @copyright Jpresta
*    @license   See the license of this module in file LICENSE.txt, thank you.
*/

/**
 * Renders a number inside the given container as a set of digit elements.
 * Each digit is wrapped in a <div class="digit"><span>...</span></div> structure.
 *
 * @param {HTMLElement} container - The HTML element where digits will be rendered.
 * @param {number|string} number - The number to display.
 */
function renderNumber(container, number) {
    const str = String(number);
    container.innerHTML = "";

    str.split("").forEach(digit => {
        const div = document.createElement("div");
        div.className = "digit";
        const span = document.createElement("span");
        span.textContent = digit;
        div.appendChild(span);
        container.appendChild(div);
    });
}

/**
 * Animates a single digit flip transition from its old value to a new value.
 *
 * @param {HTMLElement} digitElement - The .digit element to animate.
 * @param {string} newValue - The new digit value to display.
 * @param {number} delay - Delay in milliseconds before the flip animation starts.
 */
function flipDigit(digitElement, newValue, delay) {
    const span = digitElement.querySelector("span");
    setTimeout(() => {
        digitElement.classList.add("flip");
        setTimeout(() => {
            span.textContent = newValue;
        }, 120);
        setTimeout(() => {
            digitElement.classList.remove("flip");
        }, 250);
    }, delay);
}

/**
 * Updates the counter digits to reflect a new value.
 * Digits are flipped only where values differ.
 * If the number of digits changes, digits are added or removed from the start.
 *
 * @param {HTMLElement|string} container - The counter container element or its ID.
 * @param {number|string} newValue - The new number value to display.
 */
function updateCounter(container, newValue) {
    if (typeof container === "string") {
        container = document.getElementById(container);
    }
    if (!container) return;

    const oldDigits = container.querySelectorAll(".digit");
    const newStr = String(newValue).trim();
    const oldStr = Array.from(oldDigits).map(d => d.textContent.trim()).join("");

    const newArr = newStr.split("");
    const oldArr = oldStr.split("");
    const lenNew = newArr.length;
    const lenOld = oldArr.length;

    // --- Adjust digit count ---
    if (lenNew > lenOld) {
        // Add new digit elements at the beginning
        for (let i = 0; i < lenNew - lenOld; i++) {
            const div = document.createElement("div");
            div.className = "digit";
            const span = document.createElement("span");
            span.textContent = "0"; // initial value
            div.appendChild(span);
            container.insertBefore(div, container.firstChild);
        }
    } else if (lenNew < lenOld) {
        // Remove extra digits from the beginning
        for (let i = 0; i < lenOld - lenNew; i++) {
            container.removeChild(container.firstChild);
        }
    }

    // --- Flip changed digits ---
    const digits = container.querySelectorAll(".digit");
    const offset = digits.length - newArr.length;

    for (let i = newArr.length - 1; i >= 0; i--) {
        const oldVal = digits[i + offset].querySelector("span").textContent.trim();
        const newVal = newArr[i];
        if (oldVal !== newVal) {
            const delay = (newArr.length - 1 - i) * 80; // sequential flip (units → hundreds)
            flipDigit(digits[i + offset], newVal, delay);
        }
    }
}

/**
 * Observes all elements with the [data-counter] attribute.
 * Automatically renders their initial value and watches for attribute changes
 * to trigger animated counter updates.
 */
function observeCounters() {
    const counters = document.querySelectorAll("[data-counter]");

    counters.forEach(counter => {
        // Initial render
        renderNumber(counter, counter.getAttribute("data-counter"));

        // Observe attribute changes
        const observer = new MutationObserver(mutations => {
            mutations.forEach(mutation => {
                if (mutation.type === "attributes" && mutation.attributeName === "data-counter") {
                    const newValue = counter.getAttribute("data-counter");
                    updateCounter(counter, newValue);
                }
            });
        });

        observer.observe(counter, { attributes: true });
    });
}

// Auto start observer when DOM is ready
document.addEventListener("DOMContentLoaded", observeCounters);
