<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed Pack are powered by Jpresta (jpresta.com)
 *
 * @author    Jpresta
 * @copyright Jpresta
 * @license   See the license of this module in file LICENSE.txt
 */

namespace JPresta\Doctor\Service;

if (!defined('_PS_VERSION_')) {
    exit;
}

use JPresta\Doctor\Domain\Control;
use JPresta\Doctor\Domain\Problem;
use JPresta\Doctor\Exception\TimeoutException;
use JPresta\Doctor\Problem\AbstractProblem;

class DoctorService
{
    /**
     * Run a control by its id
     *
     * @param \Module $moduleInstance
     * @param int $id_control
     *
     * @return int Number of problems found
     *
     * @throws \PrestaShopDatabaseException
     * @throws \PrestaShopException
     * @throws TimeoutException
     */
    public static function runControlById($moduleInstance, $id_control, $timeoutSeconds = 0)
    {
        $currentControl = new Control($id_control);
        if (!\Validate::isLoadedObject($currentControl)) {
            throw new \PrestaShopException("id_control=$id_control not found");
        }
        $controlInstance = $currentControl->getControlInstance();
        $context = static::getLastTimeoutContext($id_control);
        if (!$context) {
            // Delete previous problems on the first start
            Problem::deleteByIdControl($currentControl->id_control);
            $currentControl->problem_fixed = 0;
            $currentControl->problem_to_fix = 0;
            $currentControl->problem_cannot_fix = 0;
            $currentControl->problem_error = 0;
        }
        try {
            $controlInstance->runWithTimeout($moduleInstance, $timeoutSeconds, static::getLastTimeoutContext($id_control), function ($problem) use ($currentControl) {
                if ($problem instanceof Problem) {
                    $problem->id_control = $currentControl->id_control;
                    $problem->add();
                    switch ($problem->state) {
                        case Problem::STATE_FIXED:
                            $currentControl->problem_fixed++;
                            break;
                        case Problem::STATE_TO_FIX:
                            $currentControl->problem_to_fix++;
                            break;
                        case Problem::STATE_CANNOT_FIX:
                            $currentControl->problem_cannot_fix++;
                            break;
                        case Problem::STATE_ERROR_IN_FIX:
                        default:
                            $currentControl->problem_error++;
                            break;
                    }
                }
            });
            $currentControl->last_run = date('Y-m-d H:i:s', time());
            $currentControl->update();
        } catch (TimeoutException $e) {
            // Save context informations so we can resume the treatment
            static::setLastTimeoutContext($id_control, $e->getContextInfos());
            // Save counts, not last run date
            $currentControl->update();
            // Re-throw the exception So the controller will re-run this control
            throw $e;
        }

        return $currentControl->problem_to_fix + $currentControl->problem_cannot_fix + $currentControl->problem_error;
    }

    /**
     * Fix a problem by its id
     *
     * @param \Module $module
     * @param int $id_problem
     * @param int $timeoutSeconds
     *
     * @return array Messages
     *
     * @throws TimeoutException Thrown if timeoutSeconds is not 0 and the problem is not fixed in the given time
     * @throws \PrestaShopException
     * @throws \PrestaShopDatabaseException
     */
    public static function fixProblemById($module, $id_problem, $timeoutSeconds = 0)
    {
        $currentProblem = new Problem($id_problem);
        if (!\Validate::isLoadedObject($currentProblem)) {
            throw new \PrestaShopException("id_problem=$id_problem not found");
        }
        $problemInstance = self::getProblemInstance($currentProblem->problem_class, $currentProblem->problem_config);
        $currentProblem->state = $problemInstance->runWithTimeout($module, $timeoutSeconds, $currentProblem->total_count);
        $messages = $problemInstance->getMessages();
        $currentProblem->problem_messages_json = json_encode($messages, JSON_UNESCAPED_UNICODE);
        $currentProblem->update();

        return $messages;
    }

    private static function getProblemInstance($className, $config)
    {
        $fullClassName = 'JPresta\\Doctor\\Problem\\' . $className;
        if (!class_exists($fullClassName)) {
            throw new \RuntimeException("Class $className not found");
        }
        if (!is_subclass_of($fullClassName, AbstractProblem::class)) {
            throw new \RuntimeException("Class $fullClassName must extend AbstractProblem");
        }
        $problem = new $fullClassName(\Db::getInstance()->connect(), _DB_PREFIX_, $config);
        $problem->init();

        return $problem;
    }

    /**
     * @throws \PrestaShopException
     * @throws \PrestaShopDatabaseException
     */
    public static function resetControls($module)
    {
        Problem::deleteAll();
        Control::deleteAll();

        return Control::updateControls($module, _PS_MODULE_DIR_ . $module->name . '/data/doctor-controls.json');
    }

    /**
     * @throws \PrestaShopException
     * @throws \PrestaShopDatabaseException
     */
    public static function updateControls($module)
    {
        return Control::updateControls($module, _PS_MODULE_DIR_ . $module->name . '/data/doctor-controls.json');
    }

    /**
     * Save the last context for a given control.
     *
     * @param int $id_control The control ID
     * @param array $contextInfos Contextual informations as array
     *
     * @return bool True on success, false otherwise
     */
    protected static function setLastTimeoutContext($id_control, $contextInfos)
    {
        if (!defined('_PS_CACHE_DIR_')) {
            return false;
        }

        $file = _PS_CACHE_DIR_ . 'last_context_' . $id_control . '.json';

        $data = [
            'id_control' => $id_control,
            'timestamp' => time(),
            'context' => $contextInfos,
        ];

        return (bool) file_put_contents($file, json_encode($data));
    }

    /**
     * Get the last saved context for a given control (if it is less than 2 minutes old).
     *
     * @param int $id_control The control ID
     *
     * @return array|null The context array or null if not found/expired
     */
    protected static function getLastTimeoutContext($id_control)
    {
        if (!defined('_PS_CACHE_DIR_')) {
            return null;
        }

        $file = _PS_CACHE_DIR_ . 'last_context_' . $id_control . '.json';

        if (!file_exists($file)) {
            return null;
        }

        $data = json_decode(file_get_contents($file), true);
        if (empty($data) || !isset($data['timestamp']) || !isset($data['context'])) {
            return null;
        }

        // Check if less than 2 minutes old
        if (time() - (int) $data['timestamp'] > 120) {
            return null;
        }

        return $data['context'];
    }
}
