<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
 *
 *    @author    Jpresta
 *    @copyright Jpresta
 *    @license   See the license of this module in file LICENSE.txt, thank you.
 */

namespace JPresta\Doctor\Domain;

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once __DIR__ . '/Control.php';
require_once __DIR__ . '/../../classes/JprestaUtils.php';

use JPresta\SpeedPack\JprestaUtils;

class Problem extends \ObjectModel
{
    const TABLE = 'jpresta_doctor_problems';

    /** @var int */
    public $id_problem;

    /** @var int */
    public $id_control;

    /** @var string */
    public $problem_class;

    /** @var string */
    public $problem_description;

    /** @var string */
    public $problem_config;

    /** @var string */
    public $problem_messages_json;

    /** @var int By default 1 but some problems can solve multiple problems so we need to count them to display a percentage */
    public $total_count = 1;

    /** @var int 1 if this Problem can be fixed by a cron job */
    public $cron = 0;

    /** @var int */
    public $state;

    // Enum-like states
    const STATE_TO_FIX = 0;
    const STATE_CANNOT_FIX = 1;
    const STATE_FIXED = 2;
    const STATE_ERROR_IN_FIX = 3;

    /**
     * PrestaShop ObjectModel definition
     */
    public static $definition = [
        'table' => 'jpresta_doctor_problems',
        'primary' => 'id_problem',
        'fields' => [
            'id_control' => ['type' => self::TYPE_INT, 'required' => true],
            'problem_class' => ['type' => self::TYPE_STRING, 'required' => true, 'size' => 255],
            'problem_description' => ['type' => self::TYPE_HTML, 'required' => false],
            'problem_config' => ['type' => self::TYPE_HTML, 'required' => false],
            'problem_messages_json' => ['type' => self::TYPE_HTML, 'required' => false],
            'total_count' => ['type' => self::TYPE_INT, 'required' => true, 'validate' => 'isUnsignedInt', 'default' => 1],
            'cron' => ['type' => self::TYPE_BOOL, 'required' => true, 'validate' => 'isBool', 'default' => '0'],
            'state' => ['type' => self::TYPE_INT, 'required' => true, 'validate' => 'isUnsignedInt'],
        ],
    ];

    public static function createTable()
    {
        JprestaUtils::dbExecuteSQL(
            'CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . self::TABLE . '` (
                `id_problem` INT UNSIGNED NOT NULL AUTO_INCREMENT,
                `id_control` INT UNSIGNED NOT NULL,
                `problem_class` VARCHAR(255) NOT NULL,
                `problem_description` TEXT NULL,
                `problem_config` TEXT NULL,
                `problem_messages_json` TEXT NULL,
                `total_count` INT UNSIGNED NOT NULL DEFAULT 1,
                `cron` TINYINT(1) NOT NULL DEFAULT 0,
                `state` TINYINT(1) UNSIGNED NOT NULL DEFAULT 0,
                PRIMARY KEY (`id_problem`),
                KEY `id_control` (`id_control`)
            ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8;',
            true,
            true
        );
    }

    public static function dropTable()
    {
        JprestaUtils::dbExecuteSQL('DROP TABLE IF EXISTS `' . _DB_PREFIX_ . self::TABLE . '`;');
    }

    public static function deleteByIdControl($id_control)
    {
        JprestaUtils::dbExecuteSQL('DELETE FROM `' . _DB_PREFIX_ . self::TABLE . '` WHERE id_control=' . (int) $id_control);
        self::updateProblemCount($id_control, 0, 0, 0, 0);
    }

    public function update($null_values = false)
    {
        $ret = parent::update($null_values);
        self::recalculateProblemCounts($this->id_control);

        return $ret;
    }

    public function add($auto_date = true, $null_values = false)
    {
        $ret = parent::add($auto_date, $null_values);
        self::recalculateProblemCounts($this->id_control);

        return $ret;
    }

    public function delete()
    {
        $ret = parent::delete();
        self::recalculateProblemCounts($this->id_control);

        return $ret;
    }

    public static function deleteAll()
    {
        JprestaUtils::dbExecuteSQL('DELETE FROM `' . _DB_PREFIX_ . self::TABLE . '`');
    }

    /**
     * Recalculate problem counters for a given control.
     *
     * @param int $id_control
     *
     * @throws \PrestaShopDatabaseException
     */
    private static function recalculateProblemCounts($id_control)
    {
        $countsRows = JprestaUtils::dbSelectRows(
            'SELECT `state`, COUNT(*) AS total
         FROM `' . _DB_PREFIX_ . self::TABLE . '`
         WHERE id_control = ' . (int) $id_control . '
         GROUP BY `state`'
        );

        // Initialize counters
        $count_fixed = 0;
        $count_to_fix = 0;
        $count_cannot_fix = 0;
        $count_error = 0;

        // Map states to counters
        foreach ($countsRows as $row) {
            switch ((int) $row['state']) {
                case Problem::STATE_FIXED:
                    $count_fixed = (int) $row['total'];
                    break;
                case Problem::STATE_TO_FIX:
                    $count_to_fix = (int) $row['total'];
                    break;
                case Problem::STATE_CANNOT_FIX:
                    $count_cannot_fix = (int) $row['total'];
                    break;
                case Problem::STATE_ERROR_IN_FIX:
                    $count_error = (int) $row['total'];
                    break;
            }
        }

        self::updateProblemCount($id_control, $count_fixed, $count_to_fix, $count_cannot_fix, $count_error);
    }

    private static function updateProblemCount($id_control, $count_fixed, $count_to_fix, $count_cannot_fix, $count_error)
    {
        $control = new Control($id_control);
        $control->problem_fixed = $count_fixed;
        $control->problem_to_fix = $count_to_fix;
        $control->problem_cannot_fix = $count_cannot_fix;
        $control->problem_error = $count_error;
        $control->update();
    }

    public static function getCountByState($state, $id_control = null)
    {
        $psVersionInt = JprestaUtils::psVersionToInt(_PS_VERSION_);
        if ($id_control) {
            return (int) JprestaUtils::dbGetValue(
                'SELECT COUNT(*) FROM `' . _DB_PREFIX_ . self::TABLE . '` p
                LEFT JOIN `' . _DB_PREFIX_ . Control::TABLE . '` c ON p.id_control=c.id_control
                WHERE p.id_control=' . (int) $id_control . '
                AND p.state=' . (int) $state . '
                AND c.active<>0
                AND (c.ps_version_min <= ' . $psVersionInt . ' AND c.ps_version_max >= ' . $psVersionInt . ')'
            );
        } else {
            return (int) JprestaUtils::dbGetValue(
                'SELECT COUNT(*) FROM `' . _DB_PREFIX_ . self::TABLE . '` p
                LEFT JOIN `' . _DB_PREFIX_ . Control::TABLE . '` c ON p.id_control=c.id_control
                WHERE p.state=' . (int) $state . '
                AND c.active<>0
                AND (c.ps_version_min <= ' . $psVersionInt . ' AND c.ps_version_max >= ' . $psVersionInt . ')'
            );
        }
    }

    public static function getCountNotFixed()
    {
        $psVersionInt = JprestaUtils::psVersionToInt(_PS_VERSION_);

        return (int) JprestaUtils::dbGetValue(
            'SELECT COUNT(*) FROM `' . _DB_PREFIX_ . self::TABLE . '` p
            LEFT JOIN `' . _DB_PREFIX_ . Control::TABLE . '` c ON p.id_control=c.id_control
            WHERE p.state<>' . self::STATE_FIXED . '
            AND c.active<>0
            AND (c.ps_version_min <= ' . $psVersionInt . ' AND c.ps_version_max >= ' . $psVersionInt . ')'
        );
    }

    public static function getFirst($id_control = null)
    {
        $psVersionInt = JprestaUtils::psVersionToInt(_PS_VERSION_);
        if (!$id_control) {
            return (int) JprestaUtils::dbGetValue(
                'SELECT id_problem FROM `' . _DB_PREFIX_ . self::TABLE . '` p
                LEFT JOIN `' . _DB_PREFIX_ . Control::TABLE . '` c ON p.id_control=c.id_control
                WHERE p.state=' . self::STATE_TO_FIX . '
                AND c.active<>0
                AND (c.ps_version_min <= ' . $psVersionInt . ' AND c.ps_version_max >= ' . $psVersionInt . ')
                ORDER BY p.id_problem ASC LIMIT 1'
            );
        } else {
            return (int) JprestaUtils::dbGetValue(
                'SELECT id_problem FROM `' . _DB_PREFIX_ . self::TABLE . '` p
                LEFT JOIN `' . _DB_PREFIX_ . Control::TABLE . '` c ON p.id_control=c.id_control
                WHERE p.state=' . self::STATE_TO_FIX . '
                AND p.id_control=' . (int) $id_control . '
                AND c.active<>0
                AND (c.ps_version_min <= ' . $psVersionInt . ' AND c.ps_version_max >= ' . $psVersionInt . ')
                ORDER BY p.id_problem ASC LIMIT 1'
            );
        }
    }

    public static function getNext($id_problem, $id_control = null)
    {
        $psVersionInt = JprestaUtils::psVersionToInt(_PS_VERSION_);
        if (!$id_control) {
            return (int) JprestaUtils::dbGetValue(
                'SELECT id_problem FROM `' . _DB_PREFIX_ . self::TABLE . '` p
                LEFT JOIN `' . _DB_PREFIX_ . Control::TABLE . '` c ON p.id_control=c.id_control
                WHERE p.id_problem>' . (int) $id_problem . '
                AND p.state=' . self::STATE_TO_FIX . '
                AND c.active<>0
                AND (c.ps_version_min <= ' . $psVersionInt . ' AND c.ps_version_max >= ' . $psVersionInt . ')
                ORDER BY p.id_problem ASC LIMIT 1'
            );
        } else {
            return (int) JprestaUtils::dbGetValue(
                'SELECT id_problem FROM `' . _DB_PREFIX_ . self::TABLE . '` p
                LEFT JOIN `' . _DB_PREFIX_ . Control::TABLE . '` c ON p.id_control=c.id_control
                WHERE p.id_problem>' . (int) $id_problem . '
                AND p.state=' . self::STATE_TO_FIX . '
                AND p.id_control=' . (int) $id_control . '
                AND c.active<>0
                AND (c.ps_version_min <= ' . $psVersionInt . ' AND c.ps_version_max >= ' . $psVersionInt . ')
                ORDER BY p.id_problem ASC LIMIT 1'
            );
        }
    }

    public static function getAllIdsForCron()
    {
        $rows = JprestaUtils::dbSelectRows('SELECT id_problem FROM `' . _DB_PREFIX_ . self::TABLE . '` WHERE cron=1 AND state=' . (int) Problem::STATE_TO_FIX, true, true);

        return array_map(function ($item) {
            return $item['id_problem'];
        }, $rows);
    }
}
