<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
 *
 *    @author    Jpresta
 *    @copyright Jpresta
 *    @license   See the license of this module in file LICENSE.txt, thank you.
 */

namespace JPresta\Doctor\Domain;

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once __DIR__ . '/AbstractControl.php';
require_once __DIR__ . '/../../classes/JprestaUtils.php';

// Automatically include Control so classe_exists() will work
foreach (glob(__DIR__ . '/../Control/*.php') as $filename) {
    if (basename($filename) !== 'index.php') {
        require_once $filename;
    }
}

use JPresta\Doctor\Control\AbstractControl;
use JPresta\SpeedPack\JprestaUtils;

class Control extends \ObjectModel
{
    const TABLE = 'jpresta_doctor_controls';

    /** @var int */
    public $id_control;

    /** @var string */
    public $uuid;

    /** @var string */
    public $control_class;

    /** @var string */
    public $control_name;

    /** @var string */
    public $control_description;

    /** @var string */
    public $control_how_to_fix;

    /** @var string */
    public $control_config;

    /** @var int */
    public $active = 1;

    /** @var int 1 if this Control can be run as a cron job */
    public $cron = 0;

    /** @var \DateTime|string */
    public $last_run;

    /** @var int */
    public $ps_version_min;

    /** @var int */
    public $ps_version_max;

    /** @var int */
    public $problem_fixed = 0;

    /** @var int */
    public $problem_to_fix = 0;

    /** @var int */
    public $problem_cannot_fix = 0;

    /** @var int */
    public $problem_error = 0;

    /**
     * PrestaShop ObjectModel definition
     */
    public static $definition = [
        'table' => self::TABLE,
        'primary' => 'id_control',
        'fields' => [
            'uuid' => ['type' => self::TYPE_STRING, 'required' => true, 'size' => 36],
            'control_class' => ['type' => self::TYPE_STRING, 'required' => true, 'size' => 255],
            'control_name' => ['type' => self::TYPE_STRING, 'required' => true, 'size' => 255],
            'control_description' => ['type' => self::TYPE_STRING, 'required' => false],
            'control_how_to_fix' => ['type' => self::TYPE_STRING, 'required' => false],
            'control_config' => ['type' => self::TYPE_STRING, 'required' => false],
            'active' => ['type' => self::TYPE_BOOL, 'required' => true, 'validate' => 'isBool', 'default' => '1'],
            'cron' => ['type' => self::TYPE_BOOL, 'required' => true, 'validate' => 'isBool', 'default' => '0'],
            'ps_version_min' => ['type' => self::TYPE_STRING, 'required' => true, 'size' => 16],
            'ps_version_max' => ['type' => self::TYPE_STRING, 'required' => true, 'size' => 16],
            'last_run' => ['type' => self::TYPE_DATE, 'required' => false, 'validate' => 'isDate', 'default' => null, 'allow_null' => true],
            'problem_fixed' => ['type' => self::TYPE_INT, 'required' => true, 'validate' => 'isUnsignedInt', 'default' => 0],
            'problem_to_fix' => ['type' => self::TYPE_INT, 'required' => true, 'validate' => 'isUnsignedInt', 'default' => 0],
            'problem_cannot_fix' => ['type' => self::TYPE_INT, 'required' => true, 'validate' => 'isUnsignedInt', 'default' => 0],
            'problem_error' => ['type' => self::TYPE_INT, 'required' => true, 'validate' => 'isUnsignedInt', 'default' => 0],
        ],
    ];

    public static function createTable()
    {
        JprestaUtils::dbExecuteSQL('CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . self::TABLE . '` (
          `id_control` INT UNSIGNED NOT NULL AUTO_INCREMENT,
          `uuid` CHAR(36) NOT NULL,
          `control_class` VARCHAR(255) NOT NULL,
          `control_name` VARCHAR(255) NOT NULL,
          `control_description` TEXT NULL,
          `control_how_to_fix` TEXT NULL,
          `control_config` TEXT NULL,
          `active` TINYINT(1) NOT NULL DEFAULT 1,
          `cron` TINYINT(1) NOT NULL DEFAULT 0,
          `ps_version_min` INT UNSIGNED NOT NULL DEFAULT 0,
          `ps_version_max` INT UNSIGNED NOT NULL DEFAULT 4294967295,
          `last_run` DATETIME NULL,
          `problem_fixed` INT UNSIGNED NOT NULL DEFAULT 0,
          `problem_to_fix` INT UNSIGNED NOT NULL DEFAULT 0,
          `problem_cannot_fix` INT UNSIGNED NOT NULL DEFAULT 0,
          `problem_error` INT UNSIGNED NOT NULL DEFAULT 0,
          PRIMARY KEY (`id_control`),
          UNIQUE KEY `uuid` (`uuid`)
        ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8;', true, true);
    }

    public static function dropTable()
    {
        JprestaUtils::dbExecuteSQL('DROP TABLE IF EXISTS `' . _DB_PREFIX_ . self::TABLE . '`;');
    }

    public static function deleteAll()
    {
        JprestaUtils::dbExecuteSQL('DELETE FROM `' . _DB_PREFIX_ . self::TABLE . '`');
    }

    /**
     * Read the JSON file and update the database with the controls.
     * Preserves existing controls that are not in the JSON file.
     * Deletes controls that are in the JSON file but without configuration (only the uuid).
     * If a control is already present, it is updated only if one of its fields really changed.
     *
     * @param \ModuleCore $module The current PrestaShop module instance
     * @param string $jsonFile Path to the JSON file containing control definitions
     * @param bool $logChanges Optional - If true, logs will be written (default: false)
     *
     * @return array
     *
     * @throws \PrestaShopDatabaseException
     * @throws \PrestaShopException
     */
    public static function updateControls($module, $jsonFile, $logChanges = false)
    {
        // --- 1. Validate JSON file existence and readability ---
        if (!file_exists($jsonFile) || !is_readable($jsonFile)) {
            throw new \PrestaShopException('JSON file not found or not readable: ' . $jsonFile);
        }

        // --- 2. Decode JSON and validate format ---
        $data = json_decode(file_get_contents($jsonFile), true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new \PrestaShopException('Invalid JSON: ' . json_last_error_msg());
        }

        if (empty($data) || !is_array($data)) {
            throw new \PrestaShopException('No controls found in JSON file.');
        }

        // --- 3. Initialize counters ---
        $created = 0;
        $updated = 0;
        $deleted = 0;

        // --- 4. Iterate through each control definition ---
        foreach ($data as $controlData) {
            if (empty($controlData['uuid'])) {
                continue; // UUID is mandatory
            }

            $uuid = pSQL($controlData['uuid']);
            $idControl = (int) \Db::getInstance()->getValue(
                'SELECT id_control FROM `' . _DB_PREFIX_ . self::TABLE . '` WHERE uuid = "' . $uuid . '"'
            );

            // --- 5. Delete control if no control_class is defined ---
            if (empty($controlData['control_class'])) {
                if ($idControl) {
                    if ($logChanges) {
                        JprestaUtils::addLog('Control ' . $uuid . ' deleted (missing class in JSON).', 1);
                    }
                    \Db::getInstance()->delete(self::TABLE, 'id_control = ' . $idControl);
                    ++$deleted;
                }
                continue;
            }

            $control = $idControl ? new self($idControl) : new self();

            // --- 6. Save current database state before modification ---
            $before = $idControl ? get_object_vars($control) : [];

            // --- 7. Assign new values from JSON file ---
            $control->uuid = $uuid;
            $control->control_class = $controlData['control_class'];
            $control->control_config = isset($controlData['control_config']) ? $controlData['control_config'] : '';
            $control->ps_version_min = isset($controlData['ps_version_min']) ? (int) $controlData['ps_version_min'] : 0;
            $control->ps_version_max = isset($controlData['ps_version_max']) ? (int) $controlData['ps_version_max'] : 4294967295;
            $control->cron = isset($controlData['cron']) ? (int) $controlData['cron'] : 0;

            // --- 8. Try to instantiate the control class ---
            try {
                $controlInstance = $control->getControlInstance();
            } catch (\Exception $e) {
                if ($idControl) {
                    if ($logChanges) {
                        JprestaUtils::addLog(
                            'Error loading control ' . $control->control_class . ': ' . $e->getMessage() . '. Control deleted.',
                            2
                        );
                    }
                    \Db::getInstance()->delete(self::TABLE, 'id_control = ' . $idControl);
                    ++$deleted;
                }
                continue;
            } catch (\Error $e) {
                if ($idControl) {
                    if ($logChanges) {
                        JprestaUtils::addLog(
                            'Error loading control ' . $control->control_class . ': ' . $e->getMessage() . '. Control deleted.',
                            2
                        );
                    }
                    \Db::getInstance()->delete(self::TABLE, 'id_control = ' . $idControl);
                    ++$deleted;
                }
                continue;
            }

            // --- 9. Generate name and description dynamically or from JSON ---
            $control->control_name = method_exists($controlInstance, 'generateName')
                ? $controlInstance->generateName($module)
                : (isset($controlData['control_name']) ? $controlData['control_name'] : '');

            $control->control_description = method_exists($controlInstance, 'generateDescription')
                ? $controlInstance->generateDescription($module)
                : (isset($controlData['control_description']) ? $controlData['control_description'] : '');

            $control->control_how_to_fix = method_exists($controlInstance, 'generateHowToFix')
                ? $controlInstance->generateHowToFix($module)
                : (isset($controlData['control_how_to_fix']) ? $controlData['control_how_to_fix'] : '');

            // --- 10. Detect if any field really changed ---
            $changed = !$idControl;
            if ($idControl) {
                $after = get_object_vars($control);
                $fieldsToCheck = [
                    'control_class',
                    'control_config',
                    'control_name',
                    'control_description',
                    'control_how_to_fix',
                    'ps_version_min',
                    'ps_version_max',
                    'cron',
                ];

                foreach ($fieldsToCheck as $field) {
                    $beforeValue = isset($before[$field]) ? (string) $before[$field] : '';
                    $afterValue = isset($after[$field]) ? (string) $after[$field] : '';
                    if ($beforeValue !== $afterValue) {
                        $changed = true;
                        break;
                    }
                }
            }

            // --- 11. Skip save if no real modification detected ---
            if (!$changed) {
                continue;
            }

            // --- 12. Reset problems when a control changes ---
            if ($idControl) {
                $control->last_run = null;
                Problem::deleteByIdControl($idControl);
            }

            // --- 13. Save control and update counters ---
            if ($control->save()) {
                if ($idControl) {
                    ++$updated;
                    if ($logChanges) {
                        JprestaUtils::addLog('Control updated: ' . $control->control_name . ' (' . $uuid . ')', 1);
                    }
                } else {
                    ++$created;
                    if ($logChanges) {
                        JprestaUtils::addLog('New control added: ' . $control->control_name . ' (' . $uuid . ')', 1);
                    }
                }
            }

            unset($control);
        }

        // --- 14. Return summary ---
        return [
            'created' => $created,
            'updated' => $updated,
            'deleted' => $deleted,
        ];
    }

    public static function exists($uuid)
    {
        return (int) JprestaUtils::dbGetValue(
            'SELECT EXISTS(
                SELECT 1
                FROM `' . _DB_PREFIX_ . self::TABLE . '`
                WHERE uuid=' . JprestaUtils::dbToString(\Db::getInstance(), $uuid) . ')'
        ) > 0;
    }

    public function delete()
    {
        \Db::getInstance()->delete(
            Problem::TABLE,
            'id_control = ' . (int) $this->id
        );

        return parent::delete();
    }

    public static function getCountActive()
    {
        $psVersionInt = JprestaUtils::psVersionToInt(_PS_VERSION_);

        return (int) JprestaUtils::dbGetValue(
            'SELECT COUNT(*) FROM `' . _DB_PREFIX_ . self::TABLE . '`
            WHERE active<>0 AND (ps_version_min <= ' . $psVersionInt . ' AND ps_version_max >= ' . $psVersionInt . ')'
        );
    }

    public static function getFirst()
    {
        $psVersionInt = JprestaUtils::psVersionToInt(_PS_VERSION_);

        return (int) JprestaUtils::dbGetValue(
            'SELECT id_control FROM `' . _DB_PREFIX_ . self::TABLE . '`
            WHERE active<>0 AND (ps_version_min <= ' . $psVersionInt . ' AND ps_version_max >= ' . $psVersionInt . ')
            ORDER BY id_control ASC LIMIT 1'
        );
    }

    public static function getNext($id_control)
    {
        $psVersionInt = JprestaUtils::psVersionToInt(_PS_VERSION_);

        return (int) JprestaUtils::dbGetValue(
            'SELECT id_control FROM `' . _DB_PREFIX_ . self::TABLE . '`
            WHERE id_control>' . (int) $id_control . ' AND active<>0
            AND (ps_version_min <= ' . $psVersionInt . ' AND ps_version_max >= ' . $psVersionInt . ')
            ORDER BY id_control ASC LIMIT 1'
        );
    }

    /**
     * @throws \PrestaShopDatabaseException
     */
    public static function getAllIdsForCron()
    {
        $rows = JprestaUtils::dbSelectRows('SELECT id_control FROM `' . _DB_PREFIX_ . self::TABLE . '` WHERE cron=1', true, true);

        return array_map(function ($item) {
            return $item['id_control'];
        }, $rows);
    }

    public static function getLastFullCheckAsString()
    {
        $psVersionInt = JprestaUtils::psVersionToInt(_PS_VERSION_);
        $raw = JprestaUtils::dbGetValue(
            'SELECT MIN(last_run) FROM `' . _DB_PREFIX_ . self::TABLE . '`
            WHERE active<>0 AND (ps_version_min <= ' . $psVersionInt . ' AND ps_version_max >= ' . $psVersionInt . ')'
        );
        if ($raw === null) {
            return null;
        }
        $lastRun = new \DateTime($raw);

        return $lastRun->format('Y-m-d H:i:s');
    }

    public function getControlInstance()
    {
        $fullClassName = 'JPresta\\Doctor\\Control\\' . $this->control_class;

        if (!class_exists($fullClassName)) {
            throw new \RuntimeException('Class ' . $fullClassName . ' not found');
        }
        if (!is_subclass_of($fullClassName, AbstractControl::class)) {
            throw new \RuntimeException('Class ' . $fullClassName . ' must extend AbstractControl');
        }

        $control = new $fullClassName(\Db::getInstance()->connect(), _DB_PREFIX_, $this->control_config);
        $control->init();

        return $control;
    }
}
