<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
 *
 *    @author    Jpresta
 *    @copyright Jpresta
 *    @license   See the license of this module in file LICENSE.txt, thank you.
 */

namespace JPresta\Doctor\Problem;

use JPresta\Doctor\Exception\TimeoutException;

if (!defined('_PS_VERSION_')) {
    exit;
}

abstract class AbstractProblem
{
    /** @var array<string, string[]> */
    private $messages = [
        'info' => [],
        'warning' => [],
        'success' => [],
        'error' => [],
    ];

    /** @var string */
    protected $config_string;

    /** @var string */
    protected $prefix;

    /** @var \PDO */
    protected $pdo;

    /** @var float Start time of the run */
    private $runStart;

    /** @var int Timeout in seconds */
    private $runTimeoutSeconds;

    /**
     * @param \PDO $pdo
     * @param string $prefix
     * @param string $config_string
     */
    public function __construct($pdo, $prefix, $config_string)
    {
        $this->config_string = $config_string;
        $this->pdo = $pdo;
        $this->prefix = $prefix;
    }

    public function init()
    {
    }

    /**
     * @param \ModuleCore $module Useful to get translations
     * @param int $total_count Initial number of problem to solve (helps to display a percentage)
     *
     * @return int state (Problem::STATE_*)
     *
     * @throws TimeoutException Should be thrown if mustStop() returns true
     */
    abstract public function run($module, $total_count = 1);

    /**
     * @param \ModuleCore $module Useful to get translations
     * @param int $timeoutSeconds 0 for nonexpiring timeout
     * @param int $total_count Initial number of problem to solve (helps to display a percentage)
     *
     * @return int state (Problem::STATE_*)
     *
     * @throws TimeoutException Should be thrown if mustStop() returns true
     */
    public function runWithTimeout($module, $timeoutSeconds = 0, $total_count = 1)
    {
        $this->runStart = microtime(true);
        $this->runTimeoutSeconds = $timeoutSeconds;

        return $this->run($module, $total_count);
    }

    /**
     * @return bool If true, the control should stop running or the server may trigger a timeout
     */
    protected function mustStop()
    {
        if ($this->runTimeoutSeconds > 0) {
            $runDuration = microtime(true) - $this->runStart;

            return $runDuration >= $this->runTimeoutSeconds;
        }

        return false;
    }

    // -------------------------------
    // Message management
    // -------------------------------

    /** Add a success message */
    protected function addSuccess($message)
    {
        $this->messages['success'][] = $message;
    }

    /** Add an info message */
    protected function addInfo($message)
    {
        $this->messages['info'][] = $message;
    }

    /** Add a warning message */
    protected function addWarning($message)
    {
        $this->messages['warning'][] = $message;
    }

    /** Add an error message */
    protected function addError($message)
    {
        $this->messages['error'][] = $message;
    }

    /**
     * Get all messages (grouped by type)
     *
     * @return array<string, string[]>
     */
    public function getMessages()
    {
        return $this->messages;
    }

    /**
     * Get messages of a specific type
     *
     * @param string $type one of 'success', 'info', 'warning', 'error'
     *
     * @return string[]
     */
    public function getMessagesByType($type)
    {
        return isset($this->messages[$type]) ? $this->messages[$type] : [];
    }
}
