<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
 *
 *    @author    Jpresta
 *    @copyright Jpresta
 *    @license   See the license of this module in file LICENSE.txt, thank you.
 */

namespace JPresta\Doctor\Control;

use JPresta\Doctor\Exception\TimeoutException;
use JPresta\SpeedPack\JprestaUtils;

if (!defined('_PS_VERSION_')) {
    exit;
}

abstract class AbstractControl
{
    /** @var string */
    protected $config_string;

    /** @var string */
    protected $prefix;

    /** @var \PDO */
    protected $pdo;

    /** @var float Start time of the run */
    private $runStart;

    /** @var int Timeout in seconds */
    private $runTimeoutSeconds;

    /**
     * @param \PDO $pdo
     * @param string $prefix
     * @param string $config_string
     */
    public function __construct($pdo, $prefix, $config_string)
    {
        $this->config_string = $config_string;
        $this->pdo = $pdo;
        $this->prefix = $prefix;
    }

    public function init()
    {
    }

    /**
     * @return bool If true, the control should stop running or the server may trigger a timeout
     */
    protected function mustStop()
    {
        if ($this->runTimeoutSeconds > 0) {
            $runDuration = microtime(true) - $this->runStart;

            return $runDuration >= $this->runTimeoutSeconds;
        }

        return false;
    }

    protected function initPSConfigurationFromConfig()
    {
        // If config_string is empty or only whitespace, do nothing
        if (empty($this->config_string) || trim($this->config_string) === '') {
            return;
        }

        $configs = json_decode($this->config_string, true);

        if (!is_array($configs)) {
            // Bad JSON
            JprestaUtils::addLog('initPSConfigurationFromConfig - invalid JSON in ' . get_class($this) . ': ' . $this->config_string, 2);

            return;
        }

        foreach ($configs as $key => $defaultValue) {
            // Skip invalid or empty keys
            if (!is_string($key) || trim($key) === '') {
                JprestaUtils::addLog(
                    'initPSConfigurationFromConfig - invalid key in ' . get_class($this),
                    2
                );
                continue;
            }

            // Skip if value is an array or object
            if (is_array($defaultValue) || is_object($defaultValue)) {
                JprestaUtils::addLog(
                    'initPSConfigurationFromConfig - skipped non-scalar value for key ' . $key,
                    2
                );
                continue;
            }

            if (JprestaUtils::getConfigurationAllShop($key) === false) {
                JprestaUtils::saveConfigurationAllShop($key, (int) $defaultValue);
            }
        }
    }

    /**
     * @param \ModuleCore $module
     *
     * @return array
     */
    public static function getFormFields($module)
    {
        return [];
    }

    /**
     * @param \ModuleCore $module Useful to get translations
     * @param mixed $contextInfos Contextual informations to resume after a previous timeout
     *
     * @throws TimeoutException Should be thrown if mustStop() returns true
     */
    abstract public function run($module, $contextInfos, $onProblem);

    /**
     * @param \ModuleCore $module Useful to get translations
     * @param int $timeoutSeconds 0 for nonexpiring timeout
     * @param array $contextInfos Contextual informations to resume after a previous timeout
     * @param callable $onProblem
     *
     * @throws TimeoutException Should be thrown if mustStop() returns true
     */
    public function runWithTimeout($module, $timeoutSeconds, $contextInfos, $onProblem)
    {
        $this->runStart = microtime(true);
        $this->runTimeoutSeconds = $timeoutSeconds;

        return $this->run($module, $contextInfos, $onProblem);
    }
}
