<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
 *
 *    @author    Jpresta
 *    @copyright Jpresta
 *    @license   See the license of this module in file LICENSE.txt, thank you.
 */

namespace JPresta\Doctor\Control;

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once __DIR__ . '/../Domain/AbstractControl.php';
require_once __DIR__ . '/../Domain/Problem.php';
require_once __DIR__ . '/../../classes/JprestaUtils.php';

use JPresta\Doctor\Domain\Problem;

class PurgeUnusedImagesControl extends AbstractControl
{
    /**
     * @var string One of ImageChecker::TYPE_* constants
     */
    private $imageType;

    /**
     * @param \PDO $pdo
     * @param string $prefix
     * @param string $config_string
     */
    public function __construct($pdo, $prefix, $config_string)
    {
        parent::__construct($pdo, $prefix, $config_string);
    }

    /**
     * Initialize the control by parsing the JSON config.
     */
    public function init()
    {
        $configValues = json_decode($this->config_string, true);
        $this->imageType = isset($configValues['image_type']) ? $configValues['image_type'] : null;
        if (!in_array($this->imageType, [ImageChecker::TYPE_PRODUCT, ImageChecker::TYPE_CATEGORY, ImageChecker::TYPE_MANUFACTURER])) {
            throw new \InvalidArgumentException('Invalid image type: ' . $this->imageType);
        }
    }

    /**
     * @param \ModuleCore $module
     *
     * @return string
     */
    public function generateName($module)
    {
        return $module->l('Check for images of type "%s" that still exist on the hard drive but are no longer referenced in the database', 'purgeunusedimagescontrol', [$this->imageType]);
    }

    /**
     * @param \ModuleCore $module
     *
     * @return string
     */
    public function generateDescription($module)
    {
        return $module->l('Delete image files of type "%s" that are no longer referenced in the database', 'purgeunusedimagescontrol', [$this->imageType]);
    }

    /**
     * Generate a localized description of how to fix this problem.
     *
     * @param \ModuleCore $module The PrestaShop module instance
     *
     * @return string
     */
    public function generateHowToFix($module)
    {
        return $module->l('Delete the corresponding image files', 'purgeunusedimagescontrol');
    }

    /**
     * {@inheritDoc}
     */
    public function run($module, $contextInfos, $onProblem)
    {
        $countAndSize = ImageChecker::process($this->imageType, false);
        $imagesToDeleteCount = (int) $countAndSize['count'];
        $imagesToDeleteSize = (float) $countAndSize['sizeMB'];

        if ($imagesToDeleteCount > 0) {
            $problem = new Problem();
            $problem->problem_class = 'PurgeUnusedImagesProblem';
            $problem->problem_description = $module->l(
                '%d image files can be deleted to save %.3f MB',
                'purgeunusedimagescontrol',
                [$imagesToDeleteCount, $imagesToDeleteSize]
            );
            $problem->problem_config = json_encode(['image_type' => $this->imageType]);
            $problem->total_count = $imagesToDeleteCount;
            $problem->state = Problem::STATE_TO_FIX;
            $problem->cron = 1;

            // Save the Problem
            call_user_func($onProblem, $problem);
        }
    }
}
