<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
 *
 *    @author    Jpresta
 *    @copyright Jpresta
 *    @license   See the license of this module in file LICENSE.txt, thank you.
 */

namespace JPresta\Doctor\Control;

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once __DIR__ . '/../Domain/AbstractControl.php';
require_once __DIR__ . '/../Domain/Problem.php';
require_once __DIR__ . '/../../classes/JprestaUtils.php';

use JPresta\Doctor\Domain\Problem;
use JPresta\SpeedPack\JprestaUtils;

class PurgePageNotFoundControl extends AbstractControl
{
    /**
     * @param \PDO $pdo
     * @param string $prefix
     * @param string $config_string
     */
    public function __construct($pdo, $prefix, $config_string)
    {
        parent::__construct($pdo, $prefix, $config_string);
    }

    public function init()
    {
        parent::initPSConfigurationFromConfig();
    }

    /**
     * @param \ModuleCore $module
     *
     * @return string
     */
    public function generateName($module)
    {
        return $module->l(
            'Check for old "page not found logs"',
            'purgepagenotfoundcontrol',
            [(int) JprestaUtils::getConfigurationAllShop('JPRESTA_DOCTOR_PURGEPAGENOTFOUND_DAYS')]
        );
    }

    /**
     * @param \ModuleCore $module
     *
     * @return string
     */
    public function generateDescription($module)
    {
        return $module->l(
            'Check for "page not found" logs older than %d days',
            'purgepagenotfoundcontrol',
            [(int) JprestaUtils::getConfigurationAllShop('JPRESTA_DOCTOR_PURGEPAGENOTFOUND_DAYS')]
        );
    }

    /**
     * Generate a localized description of how to fix this problem.
     *
     * @param \ModuleCore $module The PrestaShop module instance
     *
     * @return string
     */
    public function generateHowToFix($module)
    {
        return $module->l('Old logs are deleted from table "ps_pagenotfound"', 'purgepagenotfoundcontrol');
    }

    /**
     * @param \ModuleCore $module
     *
     * @return array
     */
    public static function getFormFields($module)
    {
        $fields = [];
        $fields[] = [
            'type' => 'integer',
            'label' => $module->l('Retention period for "Page not found" logs', 'purgepagenotfoundcontrol'),
            'name' => 'JPRESTA_DOCTOR_PURGEPAGENOTFOUND_DAYS',
            'class' => 'maxwidth10rem',
            'min' => 0,
            'max' => 3000,
            'desc' => $module->l('Number of days you want to keep these logs', 'purgepagenotfoundcontrol'),
            'suffix' => $module->l('days', 'purgepagenotfoundcontrol'),
        ];

        return $fields;
    }

    /**
     * {@inheritDoc}
     */
    public function run($module, $contextInfos, $onProblem)
    {
        if (!JprestaUtils::dbTableExists($this->prefix . 'pagenotfound')) {
            return 0;
        }
        $days = (int) JprestaUtils::getConfigurationAllShop('JPRESTA_DOCTOR_PURGEPAGENOTFOUND_DAYS');
        $dateBefore = strtotime('-' . $days . ' days');
        $dateBeforeSql = date('Y-m-d H:i:s', $dateBefore);

        $query = '
            SELECT COUNT(*)
            FROM `' . bqSQL($this->prefix) . 'pagenotfound`
            WHERE date_add < \'' . pSQL($dateBeforeSql) . '\'
        ';

        $rowToDeleteCount = (int) JprestaUtils::dbGetValue($query, true, true);

        if ($rowToDeleteCount > 0) {
            $problem = new Problem();
            $problem->problem_class = 'PurgePageNotFoundProblem';
            $problem->problem_description = $module->l(
                '%d page not found logs created before %s can be deleted',
                'purgepagenotfoundcontrol',
                [$rowToDeleteCount, $dateBeforeSql]
            );
            $problem->problem_config = $dateBeforeSql;
            $problem->total_count = $rowToDeleteCount;
            $problem->cron = 1;
            $problem->state = Problem::STATE_TO_FIX;

            // Save the Problem
            call_user_func($onProblem, $problem);
        }
    }
}
