<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
 *
 *    @author    Jpresta
 *    @copyright Jpresta
 *    @license   See the license of this module in file LICENSE.txt, thank you.
 */

namespace JPresta\Doctor\Control;

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once __DIR__ . '/../Domain/AbstractControl.php';
require_once __DIR__ . '/../Domain/Problem.php';
require_once __DIR__ . '/../../classes/JprestaUtils.php';

use JPresta\Doctor\Domain\Problem;
use JPresta\SpeedPack\JprestaUtils;

class PurgeOldGuestsControl extends AbstractControl
{
    /**
     * @param \PDO $pdo
     * @param string $prefix
     * @param string $config_string
     */
    public function __construct($pdo, $prefix, $config_string)
    {
        parent::__construct($pdo, $prefix, $config_string);
    }

    public function init()
    {
        parent::initPSConfigurationFromConfig();
    }

    /**
     * @param \ModuleCore $module
     *
     * @return string
     */
    public function generateName($module)
    {
        return $module->l('Check for old guests', 'purgeoldguestscontrol');
    }

    /**
     * @param \ModuleCore $module
     *
     * @return string
     */
    public function generateDescription($module)
    {
        return $module->l(
            'Check for guests older than %d days',
            'purgeoldguestscontrol',
            [(int) JprestaUtils::getConfigurationAllShop('JPRESTA_DOCTOR_PURGEOLDGUESTS_DAYS')]
        );
    }

    /**
     * Generate a localized description of how to fix this problem.
     *
     * @param \ModuleCore $module The PrestaShop module instance
     *
     * @return string
     */
    public function generateHowToFix($module)
    {
        return $module->l('Old guests are removed from the "ps_guest" table, their carts are deleted using the Cart::delete function, and their logs are cleared from the "ps_connections", "ps_connections_page", and "ps_connections_source" tables', 'purgeoldguestscontrol');
    }

    /**
     * @param \ModuleCore $module
     *
     * @return array
     */
    public static function getFormFields($module)
    {
        $fields = [];
        $fields[] = [
            'type' => 'integer',
            'label' => $module->l('Retention period of guest visitors', 'purgeoldguestscontrol'),
            'name' => 'JPRESTA_DOCTOR_PURGEOLDGUESTS_DAYS',
            'class' => 'maxwidth10rem',
            'min' => 7,
            'max' => 3000,
            'desc' => $module->l('Number of days you want to keep guests (minimum 7)', 'purgeoldguestscontrol'),
            'suffix' => $module->l('days', 'purgeoldguestscontrol'),
        ];

        return $fields;
    }

    /**
     * {@inheritDoc}
     */
    public function run($module, $contextInfos, $onProblem)
    {
        $days = (int) JprestaUtils::getConfigurationAllShop('JPRESTA_DOCTOR_PURGEOLDGUESTS_DAYS');
        $dateBefore = strtotime('-' . $days . ' days');
        $dateBeforeSql = date('Y-m-d H:i:s', $dateBefore);

        $sqlOldGuest = '
            SELECT COUNT(*)
            FROM `' . bqSQL($this->prefix) . 'guest` g
            LEFT JOIN (
                SELECT id_guest, MAX(date_add) AS last_date
                FROM `' . bqSQL($this->prefix) . 'connections`
                GROUP BY id_guest
            ) c ON c.id_guest = g.id_guest
            WHERE c.last_date IS NULL OR c.last_date < \'' . pSQL($dateBeforeSql) . '\';';

        $rowGuestsToDeleteCount = (int) JprestaUtils::dbGetValue($sqlOldGuest, true, true);

        if ($rowGuestsToDeleteCount > 0) {
            $problem = new Problem();
            $problem->problem_class = 'PurgeOldGuestsProblem';
            $problem->problem_description = $module->l(
                '%d guests older than %s can be deleted',
                'purgeoldguestscontrol',
                [$rowGuestsToDeleteCount, $dateBeforeSql]
            );
            $problem->problem_config = $dateBeforeSql;
            $problem->total_count = $rowGuestsToDeleteCount;
            $problem->cron = 1;
            $problem->state = Problem::STATE_TO_FIX;

            // Save the Problem
            call_user_func($onProblem, $problem);
        }
    }
}
