<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
 *
 *    @author    Jpresta
 *    @copyright Jpresta
 *    @license   See the license of this module in file LICENSE.txt, thank you.
 */

namespace JPresta\Doctor\Control;

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once __DIR__ . '/../Domain/AbstractControl.php';
require_once __DIR__ . '/../Domain/Problem.php';
require_once __DIR__ . '/../../classes/JprestaUtils.php';

use JPresta\Doctor\Domain\Problem;
use JPresta\SpeedPack\JprestaUtils;

class PurgeLogsControl extends AbstractControl
{
    /**
     * @param \PDO $pdo
     * @param string $prefix
     * @param string $config_string
     */
    public function __construct($pdo, $prefix, $config_string)
    {
        parent::__construct($pdo, $prefix, $config_string);
    }

    public function init()
    {
        parent::initPSConfigurationFromConfig();
    }

    /**
     * @param \ModuleCore $module
     *
     * @return string
     */
    public function generateName($module)
    {
        return $module->l('Check for old Prestashop logs', 'purgelogscontrol');
    }

    /**
     * @param \ModuleCore $module
     *
     * @return string
     */
    public function generateDescription($module)
    {
        return $module->l(
            'Check for INFO/WARNING logs older than %d days and ERROR/CRITICAL logs older than %d days',
            'purgelogscontrol',
            [
                (int) JprestaUtils::getConfigurationAllShop('JPRESTA_DOCTOR_PURGELOGS12_DAYS'),
                (int) JprestaUtils::getConfigurationAllShop('JPRESTA_DOCTOR_PURGELOGS34_DAYS'),
            ]
        );
    }

    /**
     * Generate a localized description of how to fix this problem.
     *
     * @param \ModuleCore $module The PrestaShop module instance
     *
     * @return string
     */
    public function generateHowToFix($module)
    {
        return $module->l('Old logs are deleted from table "ps_log"', 'purgelogscontrol');
    }

    /**
     * @param \ModuleCore $module
     *
     * @return array
     */
    public static function getFormFields($module)
    {
        $fields = [];
        $fields[] = [
            'type' => 'integer',
            'label' => $module->l('Retention period for INFO/WARNING logs', 'purgelogscontrol'),
            'name' => 'JPRESTA_DOCTOR_PURGELOGS12_DAYS',
            'class' => 'maxwidth10rem',
            'min' => 0,
            'max' => 3000,
            'desc' => $module->l('Number of days you want to keep these logs', 'purgelogscontrol'),
            'suffix' => $module->l('days', 'purgelogscontrol'),
        ];
        $fields[] = [
            'type' => 'integer',
            'label' => $module->l('Retention period for ERROR/CRITICAL logs', 'purgelogscontrol'),
            'name' => 'JPRESTA_DOCTOR_PURGELOGS34_DAYS',
            'class' => 'maxwidth10rem',
            'min' => 0,
            'max' => 3000,
            'desc' => $module->l('Number of days you want to keep these logs', 'purgelogscontrol'),
            'suffix' => $module->l('days', 'purgelogscontrol'),
        ];

        return $fields;
    }

    /**
     * {@inheritDoc}
     */
    public function run($module, $contextInfos, $onProblem)
    {
        $days12 = (int) JprestaUtils::getConfigurationAllShop('JPRESTA_DOCTOR_PURGELOGS12_DAYS');
        $days34 = (int) JprestaUtils::getConfigurationAllShop('JPRESTA_DOCTOR_PURGELOGS34_DAYS');

        $dateBeforeSql12 = date('Y-m-d H:i:s', strtotime('-' . $days12 . ' days'));
        $dateBeforeSql34 = date('Y-m-d H:i:s', strtotime('-' . $days34 . ' days'));

        $query = '
            SELECT COUNT(*)
            FROM `' . bqSQL($this->prefix) . 'log`
            WHERE (severity < 3 AND date_add < \'' . pSQL($dateBeforeSql12) . '\')
               OR (severity > 2 AND date_add < \'' . pSQL($dateBeforeSql34) . '\')
        ';

        $rowToDeleteCount = (int) JprestaUtils::dbGetValue($query, true, true);

        if ($rowToDeleteCount > 0) {
            $problem = new Problem();
            $problem->problem_class = 'PurgeLogsProblem';
            $problem->problem_description = $module->l(
                '%d logs can be deleted',
                'purgelogscontrol',
                [$rowToDeleteCount]
            );

            // JSON_THROW_ON_ERROR n'existe pas en PHP 5.6
            $problem->problem_config = json_encode(
                [
                    'dateBeforeSql12' => $dateBeforeSql12,
                    'dateBeforeSql34' => $dateBeforeSql34,
                ]
            );
            $problem->total_count = $rowToDeleteCount;

            $problem->cron = 1;
            $problem->state = Problem::STATE_TO_FIX;

            // Save the Problem
            call_user_func($onProblem, $problem);
        }
    }
}
