<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
 *
 *    @author    Jpresta
 *    @copyright Jpresta
 *    @license   See the license of this module in file LICENSE.txt, thank you.
 */

namespace JPresta\Doctor\Control;

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once __DIR__ . '/../Domain/AbstractControl.php';
require_once __DIR__ . '/../Domain/Problem.php';
require_once __DIR__ . '/../../classes/JprestaUtils.php';

use JPresta\Doctor\Domain\Problem;
use JPresta\SpeedPack\JprestaUtils;

class PurgeDeletedModulesControl extends AbstractControl
{
    /**
     * @param \PDO $pdo
     * @param string $prefix
     * @param string $config_string
     */
    public function __construct($pdo, $prefix, $config_string)
    {
        parent::__construct($pdo, $prefix, $config_string);
    }

    /**
     * @param \ModuleCore $module
     *
     * @return string
     */
    public function generateName($module)
    {
        return $module->l('Check for deleted modules still existing in database', 'purgedeletedmodulescontrol');
    }

    /**
     * @param \ModuleCore $module
     *
     * @return string
     */
    public function generateDescription($module)
    {
        return $module->l('Remove all data related to modules that are no longer present on the hard drive or that cannot be instanciated', 'purgedeletedmodulescontrol');
    }

    /**
     * Generate a localized description of how to fix this problem.
     *
     * @param \ModuleCore $module The PrestaShop module instance
     *
     * @return string
     */
    public function generateHowToFix($module)
    {
        return $module->l('All rows referencing the ID of the missing or corrupted module are deleted, and its directory is archived as a ZIP file in the modules directory', 'purgedeletedmodulescontrol');
    }

    /**
     * {@inheritDoc}
     */
    public function run($module, $contextInfos, $onProblem)
    {
        $modulesNames = JprestaUtils::dbSelectRows(
            'SELECT name FROM ' . JprestaUtils::dbEscapeIdentifier($this->prefix . 'module'),
            true,
            true
        );

        foreach ($modulesNames as $moduleNameRow) {
            $moduleName = $moduleNameRow['name'];
            $moduleDir = _PS_MODULE_DIR_ . '/' . $moduleName;
            if (!is_dir($moduleDir)) {
                $problem = new Problem();
                $problem->problem_class = 'PurgeDeletedModulesProblem';
                $problem->problem_description = $module->l(
                    'Module %s does not exists anymore in %s. All data related to this module should be deleted.',
                    'purgedeletedmodulescontrol',
                    [$moduleName, $moduleDir]
                );
                $problem->problem_config = $moduleName;
                $problem->state = Problem::STATE_TO_FIX;

                // Save the Problem
                call_user_func($onProblem, $problem);
            } else {
                $moduleInstance = \Module::getInstanceByName($moduleName);
                if (!$moduleInstance) {
                    $problem = new Problem();
                    $problem->problem_class = 'PurgeDeletedModulesProblem';
                    $problem->problem_description = $module->l(
                        'Module %s does not be instanciated. The directory %s and all data related to this module should be deleted.',
                        'purgedeletedmodulescontrol',
                        [$moduleName, $moduleDir]
                    );
                    $problem->problem_config = $moduleName;
                    $problem->state = Problem::STATE_TO_FIX;

                    // Save the Problem
                    call_user_func($onProblem, $problem);
                }
            }
        }
    }
}
