<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
 *
 *    @author    Jpresta
 *    @copyright Jpresta
 *    @license   See the license of this module in file LICENSE.txt, thank you.
 */

namespace JPresta\Doctor\Control;

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once __DIR__ . '/../Domain/AbstractControl.php';
require_once __DIR__ . '/../Domain/Problem.php';
require_once __DIR__ . '/../../classes/JprestaUtils.php';

use JPresta\Doctor\Domain\Problem;
use JPresta\SpeedPack\JprestaUtils;

class PurgeConnectionLogsControl extends AbstractControl
{
    /**
     * @param \PDO $pdo
     * @param string $prefix
     * @param string $config_string
     */
    public function __construct($pdo, $prefix, $config_string)
    {
        parent::__construct($pdo, $prefix, $config_string);
    }

    public function init()
    {
        parent::initPSConfigurationFromConfig();
    }

    /**
     * @param \ModuleCore $module
     *
     * @return string
     */
    public function generateName($module)
    {
        return $module->l('Check for old connection logs', 'purgeconnectionlogscontrol');
    }

    /**
     * @param \ModuleCore $module
     *
     * @return string
     */
    public function generateDescription($module)
    {
        return $module->l(
            'Check for connection logs older than %d days',
            'purgeconnectionlogscontrol',
            [(int) JprestaUtils::getConfigurationAllShop('JPRESTA_DOCTOR_PURGECONNECTIONLOGS_DAYS')]
        );
    }

    /**
     * Generate a localized description of how to fix this problem.
     *
     * @param \ModuleCore $module The PrestaShop module instance
     *
     * @return string
     */
    public function generateHowToFix($module)
    {
        return $module->l('Old logs are deleted from tables "ps_connections", "ps_connections_page" and "ps_connections_source"', 'purgeconnectionlogscontrol');
    }

    /**
     * @param \ModuleCore $module
     *
     * @return array
     */
    public static function getFormFields($module)
    {
        $fields = [];
        $fields[] = [
            'type' => 'integer',
            'label' => $module->l('Retention period for connections logs of visitors', 'purgeconnectionlogscontrol'),
            'name' => 'JPRESTA_DOCTOR_PURGECONNECTIONLOGS_DAYS',
            'class' => 'maxwidth10rem',
            'min' => 0,
            'max' => 3000,
            'desc' => $module->l('Number of days you want to keep these logs', 'purgeconnectionlogscontrol'),
            'suffix' => $module->l('days', 'purgeconnectionlogscontrol'),
        ];

        return $fields;
    }

    /**
     * {@inheritDoc}
     */
    public function run($module, $contextInfos, $onProblem)
    {
        $days = (int) JprestaUtils::getConfigurationAllShop('JPRESTA_DOCTOR_PURGECONNECTIONLOGS_DAYS');
        $dateBefore = strtotime('-' . $days . ' days');
        $dateBeforeSql = date('Y-m-d H:i:s', $dateBefore);

        // Connections of guests are ignored because they will be deleted by the "PurgeOldGuestsControl"
        $query = '
            SELECT COUNT(*)
            FROM `' . bqSQL($this->prefix) . 'connections` c
            WHERE c.date_add < \'' . pSQL($dateBeforeSql) . '\' AND c.id_guest = 0;
        ';

        $rowToDeleteCount = (int) JprestaUtils::dbGetValue($query, true, true);

        if ($rowToDeleteCount > 0) {
            $problem = new Problem();
            $problem->problem_class = 'PurgeConnectionLogsProblem';
            $problem->problem_description = $module->l(
                '%d connections logs of visitors (IP, date, pages, origin/referer) created before %s can be deleted',
                'purgeconnectionlogscontrol',
                [$rowToDeleteCount, $dateBeforeSql]
            );
            $problem->problem_config = $dateBeforeSql;
            $problem->total_count = $rowToDeleteCount;
            $problem->cron = 1;
            $problem->state = Problem::STATE_TO_FIX;

            // Save the Problem
            call_user_func($onProblem, $problem);
        }
    }
}
