<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
 *
 *    @author    Jpresta
 *    @copyright Jpresta
 *    @license   See the license of this module in file LICENSE.txt, thank you.
 */

namespace JPresta\Doctor\Control;

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once __DIR__ . '/../Domain/AbstractControl.php';
require_once __DIR__ . '/../Domain/Problem.php';
require_once __DIR__ . '/../../classes/JprestaUtils.php';

use JPresta\Doctor\Domain\Problem;
use JPresta\SpeedPack\JprestaUtils;

class PurgeAbandonedCartsControl extends AbstractControl
{
    /**
     * @param \PDO $pdo
     * @param string $prefix
     * @param string $config_string
     */
    public function __construct($pdo, $prefix, $config_string)
    {
        parent::__construct($pdo, $prefix, $config_string);
    }

    public function init()
    {
        parent::initPSConfigurationFromConfig();
    }

    /**
     * @param \ModuleCore $module
     *
     * @return string
     */
    public function generateName($module)
    {
        return $module->l('Check for old abandoned carts', 'purgeabandonedcartscontrol');
    }

    /**
     * @param \ModuleCore $module
     *
     * @return string
     */
    public function generateDescription($module)
    {
        return $module->l(
            'Check for abandoned carts older than %d days',
            'purgeabandonedcartscontrol',
            [(int) JprestaUtils::getConfigurationAllShop('JPRESTA_DOCTOR_PURGEABANDONEDCARTS_DAYS')]
        );
    }

    /**
     * Generate a localized description of how to fix this problem.
     *
     * @param \ModuleCore $module The PrestaShop module instance
     *
     * @return string
     */
    public function generateHowToFix($module)
    {
        return $module->l('Abandoned carts are deleted using the Cart::delete function', 'purgeabandonedcartscontrol');
    }

    /**
     * @param \ModuleCore $module
     *
     * @return array
     */
    public static function getFormFields($module)
    {
        $fields = [];
        $fields[] = [
            'type' => 'integer',
            'label' => $module->l('Retention period of abandoned carts', 'purgeabandonedcartscontrol'),
            'name' => 'JPRESTA_DOCTOR_PURGEABANDONEDCARTS_DAYS',
            'class' => 'maxwidth10rem',
            'min' => 7,
            'max' => 3000,
            'desc' => $module->l('Number of days you want to keep abandoned carts (minimum 7 but we recommend at least 30)', 'purgeabandonedcartscontrol'),
            'suffix' => $module->l('days', 'purgeabandonedcartscontrol'),
        ];

        return $fields;
    }

    /**
     * {@inheritDoc}
     */
    public function run($module, $contextInfos, $onProblem)
    {
        $days = (int) JprestaUtils::getConfigurationAllShop('JPRESTA_DOCTOR_PURGEABANDONEDCARTS_DAYS');
        $dateBefore = strtotime('-' . $days . ' days');
        $dateBeforeSql = date('Y-m-d H:i:s', $dateBefore);

        $query = '
            SELECT COUNT(*)
            FROM `' . bqSQL($this->prefix) . 'cart` c
            LEFT JOIN `' . bqSQL($this->prefix) . 'orders` o ON c.id_cart = o.id_cart
            WHERE (o.id_order IS NULL OR o.id_order = 0)
            AND c.date_upd < \'' . pSQL($dateBeforeSql) . '\'';

        $rowToDeleteCount = (int) JprestaUtils::dbGetValue($query, true, true);

        if ($rowToDeleteCount > 0) {
            $problem = new Problem();
            $problem->problem_class = 'PurgeAbandonedCartsProblem';
            $problem->problem_description = $module->l(
                '%d carts older than %s can be deleted',
                'purgeabandonedcartscontrol',
                [$rowToDeleteCount, $dateBeforeSql]
            );
            $problem->problem_config = $dateBeforeSql;
            $problem->total_count = $rowToDeleteCount;
            $problem->cron = 1;
            $problem->state = Problem::STATE_TO_FIX;

            // Save the Problem
            call_user_func($onProblem, $problem);
        }
    }
}
