<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
 *
 * @author    Jpresta
 * @copyright Jpresta
 * @license   See the license of this module in file LICENSE.txt, thank you.
 */

namespace JPresta\Doctor\Control;

use JPresta\SpeedPack\JprestaUtils;

if (!defined('_PS_VERSION_')) {
    exit;
}

class ImageChecker
{
    const TYPE_PRODUCT = 'product';
    const TYPE_CATEGORY = 'category';
    const TYPE_MANUFACTURER = 'manufacturer';
    const VALID_EXTENSIONS = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'avif'];

    /**
     * Scan and optionally delete unused image files.
     *
     * @param string $type One of the TYPE_* constants
     * @param bool $delete Whether to delete the unused files
     *
     * @return array count and total size
     */
    public static function process($type, $delete = false)
    {
        $sql = 'SELECT name FROM `' . _DB_PREFIX_ . 'image_type`';
        switch ($type) {
            case ImageChecker::TYPE_PRODUCT:
                $sql .= ' WHERE products=1';
                break;
            case ImageChecker::TYPE_CATEGORY:
                $sql .= ' WHERE categories=1';
                break;
            case ImageChecker::TYPE_MANUFACTURER:
                $sql .= ' WHERE manufacturers=1';
                break;
        }
        $formatsRows = JprestaUtils::dbSelectRows($sql, true, true);
        $typeFormats = array_column($formatsRows, 'name');

        $baseDir = static::getBaseDir($type);
        if (!is_dir($baseDir)) {
            JprestaUtils::addLog('Doctor: Invalid image directory: ' . $baseDir, 2);

            return ['count' => 0, 'sizeMB' => 0];
        }

        $dirIterator = new \RecursiveDirectoryIterator($baseDir, \FilesystemIterator::SKIP_DOTS);
        $iterator = new \RecursiveIteratorIterator($dirIterator, \RecursiveIteratorIterator::LEAVES_ONLY);

        $imagesToDeleteCount = 0;
        $imagesToDeleteSize = 0;
        $lastImageId = null;
        $lastExistsInDb = false;

        foreach ($iterator as $fileInfo) {
            // LEAVES_ONLY: skip directories

            $filename = $fileInfo->getFilename();
            $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

            if (!in_array($extension, static::VALID_EXTENSIONS, true)) {
                continue;
            }

            // Match pattern: <idImage>-<ImageType/format>.<ext>
            $basename = pathinfo($filename, PATHINFO_FILENAME);
            if (!preg_match('/^([0-9]+)(?:-([a-zA-Z0-9_ -]+))?$/', $basename, $matches)) {
                // Not an image we are looking for
                continue;
            }
            $idImage = (int) $matches[1];
            $format = isset($matches[2]) ? $matches[2] : ''; // e.g. "small_default"

            $mustDelete = false;

            //
            // Check if the format is in the list of existing formats
            //
            if (!empty($format) && !in_array($format, $typeFormats, true)) {
                // This format is not in the list of existing formats
                $mustDelete = true;
            }

            //
            // Check if the ID exists in the database
            //
            if (!$mustDelete && $idImage !== $lastImageId) {
                // Only one SQL check per ID
                $lastImageId = $idImage;
                $lastExistsInDb = static::existsInDatabase($type, $idImage);
            }
            $mustDelete = $mustDelete || !$lastExistsInDb;

            if ($mustDelete) {
                ++$imagesToDeleteCount;
                $imagesToDeleteSize += $fileInfo->getSize();

                if ($delete) {
                    // TODO add a warning if the file cannot be deleted
                    JprestaUtils::deleteFile($fileInfo->getPathname());
                }
            }
        }

        return [
            'count' => $imagesToDeleteCount,
            'sizeMB' => round($imagesToDeleteSize / (1024 * 1024), 3, PHP_ROUND_HALF_UP),
        ];
    }

    /**
     * Check in database if the entity exists.
     *
     * @param string $type
     * @param int $id
     *
     * @return bool
     */
    protected static function existsInDatabase($type, $id)
    {
        switch ($type) {
            case self::TYPE_PRODUCT:
                $table = 'image';
                $column = 'id_image';
                break;
            case self::TYPE_CATEGORY:
                $table = 'category';
                $column = 'id_category';
                break;
            case self::TYPE_MANUFACTURER:
                $table = 'manufacturer';
                $column = 'id_manufacturer';
                break;
            default:
                // Avoid deleting images from other tables
                return true;
        }

        // Do not log on error because it could be a lot of logs
        $sql = 'SELECT 1 FROM `' . _DB_PREFIX_ . $table . '` WHERE `' . $column . '` = ' . (int) $id;

        return (int) JprestaUtils::dbGetValue($sql, false, false) > 0;
    }

    /**
     * Get base image directory depending on type.
     *
     * @param string $type
     *
     * @return string
     */
    protected static function getBaseDir($type)
    {
        switch ($type) {
            case self::TYPE_CATEGORY:
                return rtrim(_PS_CAT_IMG_DIR_, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR;
            case self::TYPE_MANUFACTURER:
                return rtrim(_PS_MANU_IMG_DIR_, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR;
            case self::TYPE_PRODUCT:
            default:
                if (JprestaUtils::version_compare(_PS_VERSION_, '8', '<')) {
                    return rtrim(_PS_PROD_IMG_DIR_, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR;
                } else {
                    return rtrim(_PS_PRODUCT_IMG_DIR_, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR;
                }
        }
    }
}
