<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed Pack are powered by Jpresta (jpresta.com)
 *
 * @author    Jpresta
 * @copyright Jpresta
 * @license   See the license of this module in file LICENSE.txt
 */

namespace JPresta\Doctor\Control;

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once __DIR__ . '/../Control/DbRelation.php';

/**
 * Configuration for a database relation control.
 *
 * This class defines how a relation described by {@see DbRelation} should be handled
 * when inconsistencies are detected. It includes fixability, deletion policies,
 * and mapping to PrestaShop ObjectModels for automated cleanup.
 */
final class DbRelationControlConfig
{
    /**
     * @var DbRelation The relation definition (source/target tables, columns, etc.).
     */
    public $dbRelation;

    /**
     * @var bool Whether the relation can be automatically fixed.
     *           By default, relations are considered not fixable.
     */
    public $canFix = false;

    /**
     * @var bool whether the row must be deleted if the reference is missing
     */
    public $deleteRow = false;

    /**
     * @var string|null the PrestaShop ObjectModel class to use for deletion when the reference is missing
     */
    public $deleteObjectModelClass;

    /**
     * @var bool if true, set the reference field to NULL (or zero if {@see DbRelation::$fromFieldZeroIsNull} is true)
     *           when the reference is missing
     */
    public $setNull = false;

    // -----------------------------------------------------------------
    // Builders
    // -----------------------------------------------------------------

    /**
     * Build a config instance from a JSON string.
     *
     * @param string $json JSON-encoded config
     *
     * @return DbRelationControlConfig
     *
     * @throws \InvalidArgumentException if the JSON is invalid
     */
    public static function buildFromJson($json)
    {
        $data = json_decode($json, true);
        if ($data === null) {
            throw new \InvalidArgumentException('Invalid JSON: ' . json_last_error_msg());
        }

        return self::buildFromArray($data);
    }

    /**
     * Build a config instance from an associative array.
     *
     * @param array $data the array of configuration values
     *
     * @return DbRelationControlConfig
     */
    public static function buildFromArray($data)
    {
        $controlConfig = new DbRelationControlConfig();

        foreach ($data as $key => $value) {
            if (property_exists($controlConfig, $key)) {
                if ($key === 'dbRelation') {
                    $controlConfig->dbRelation = DbRelation::buildFromArray($value);
                } else {
                    $controlConfig->$key = $value;
                }
            }
        }

        return $controlConfig;
    }
}
