<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed Pack are powered by Jpresta (jpresta.com)
 *
 * @author    Jpresta
 * @copyright Jpresta
 * @license   See the license of this module in file LICENSE.txt
 */

namespace JPresta\Doctor\Control;

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once __DIR__ . '/../Domain/AbstractControl.php';
require_once __DIR__ . '/../Domain/Problem.php';
require_once __DIR__ . '/../Control/DbRelationControlConfig.php';
require_once __DIR__ . '/../../classes/JprestaUtils.php';

use JPresta\Doctor\Domain\Problem;
use JPresta\Doctor\Exception\TimeoutException;
use JPresta\SpeedPack\JprestaUtils;
use PDO;

/**
 * Control class that validates database relations.
 *
 * This control ensures that a column in a source table correctly references
 * existing values in a target table (foreign key integrity). It can also yield
 * problems describing missing references and whether they are fixable.
 */
class DbRelationControl extends AbstractControl
{
    /**
     * @var DbRelationControlConfig parsed configuration for this control
     */
    public $config;

    /**
     * Constructor.
     *
     * @param \PDO $this->pdo PDO connection to the database
     * @param string $prefix Table prefix
     * @param string $config_string JSON string containing the control config
     */
    public function __construct($pdo, $prefix, $config_string)
    {
        parent::__construct($pdo, $prefix, $config_string);
    }

    /**
     * Initialize the control by parsing the JSON config.
     */
    public function init()
    {
        $this->config = DbRelationControlConfig::buildFromJson($this->config_string);
    }

    /**
     * Generate a localized human-readable name for this control.
     *
     * @param \ModuleCore $module The PrestaShop module instance
     *
     * @return string
     */
    public function generateName($module)
    {
        // Must be on one single line or it will not be parsed by Prestashop translation system
        return $module->l('Check references from %s.%s to %s.%s', 'dbrelationcontrol', [$this->config->dbRelation->fromTable, $this->config->dbRelation->fromField, $this->config->dbRelation->toTable, $this->config->dbRelation->toField]);
    }

    /**
     * Generate a localized description of what this control checks.
     *
     * @param \ModuleCore $module The PrestaShop module instance
     *
     * @return string
     */
    public function generateDescription($module)
    {
        // Must be on one single line or it will not be parsed by Prestashop translation system
        return $module->l('Check in database that the column "%s" of table "%s" references an existing value in column "%s" of table "%s"', 'dbrelationcontrol', [$this->config->dbRelation->fromField, $this->config->dbRelation->fromTable, $this->config->dbRelation->toField, $this->config->dbRelation->toTable]);
    }

    /**
     * Generate a localized description of how to fix this problem.
     *
     * @param \ModuleCore $module The PrestaShop module instance
     *
     * @return string
     */
    public function generateHowToFix($module)
    {
        if (!$this->config->canFix) {
            return $module->l('This issue has not yet been thoroughly analyzed for an automatic fix. Feel free to contact support for assistance in resolving it', 'dbrelationcontrol');
        }
        // Option 1: set reference to NULL (or 0)
        if ($this->config->setNull) {
            if ($this->config->dbRelation->fromFieldZeroIsNull) {
                return $module->l('The missing reference will be reset to 0 (equivalent to NULL)', 'dbrelationcontrol');
            }
            if ($this->config->dbRelation->fromField === 'id_lang') {
                return $module->l('The reference to the deleted language will be replaced with the shop’s default language', 'dbrelationcontrol');
            }

            return $module->l('The missing reference will be set to NULL', 'dbrelationcontrol');
        }
        // Option 2: delete via ObjectModel
        if ($this->config->deleteObjectModelClass) {
            return $module->l('The entity %s with this missing reference will be deleted', 'dbrelationcontrol', [$this->config->deleteObjectModelClass]);
        }
        // Option 3: direct row deletion
        if ($this->config->deleteRow) {
            return $module->l('The row with this missing reference will be deleted', 'dbrelationcontrol');
        }

        return null;
    }

    /**
     * Execute the control and yield problems for each missing reference found.
     *
     * @param \ModuleCore $module The PrestaShop module instance
     * @param array $contextInfos Start index (used to resume after an interruption)
     *
     * @return \Generator Yields Problem instances describing broken relations
     *
     * @throws TimeoutException when mustStop() returns true
     */
    public function run($module, $contextInfos, $onProblem)
    {
        $index = 0;
        $total_count = -1;
        if ($contextInfos != null && isset($contextInfos['index']) && isset($contextInfos['total_count'])) {
            $index = (int) $contextInfos['index'];
            $total_count = (int) $contextInfos['total_count'];
        }

        // Validate tables and columns existence
        if (!JprestaUtils::dbtableExists($this->prefix . $this->config->dbRelation->fromTable)
            || !JprestaUtils::dbtableExists($this->prefix . $this->config->dbRelation->toTable)
            || !JprestaUtils::dbColumnExists($this->prefix . $this->config->dbRelation->fromTable, $this->config->dbRelation->fromField)
            || !JprestaUtils::dbColumnExists($this->prefix . $this->config->dbRelation->toTable, $this->config->dbRelation->toField)) {
            JprestaUtils::addLog('Doctor | Ignoring this control as one of the table/column does not exist (but should): ' . $this->generateName($module), 2);

            return 0;
        }

        // Build safe SQL identifiers
        $fromTableSql = self::tableWithPrefixEscaped($this->config->dbRelation->fromTable, $this->prefix);
        $toTableSql = self::tableWithPrefixEscaped($this->config->dbRelation->toTable, $this->prefix);
        $fromFieldSql = self::qid($this->config->dbRelation->fromField);
        $toFieldSql = self::qid($this->config->dbRelation->toField);

        // WHERE clauses
        $whereClauses = [];
        if (!empty($this->config->dbRelation->fromFieldZeroIsNull)) {
            $whereClauses[] = sprintf('(fromTable.%1$s IS NOT NULL AND fromTable.%1$s <> 0)', $fromFieldSql);
        } else {
            $whereClauses[] = sprintf('fromTable.%s IS NOT NULL', $fromFieldSql);
        }
        if (!empty($this->config->dbRelation->deletedColumn)) {
            // Exclude soft-deleted rows
            $whereClauses[] = sprintf('fromTable.%s = 0', self::qid($this->config->dbRelation->deletedColumn));
        }

        // SQL to find orphan references
        $sqlQuery = sprintf(
            'FROM %1$s AS fromTable
         LEFT JOIN %2$s AS toTable ON fromTable.%3$s = toTable.%4$s
         WHERE %5$s AND toTable.%4$s IS NULL',
            $fromTableSql,
            $toTableSql,
            $fromFieldSql,
            $toFieldSql,
            implode(' AND ', $whereClauses)
        );
        $sqlOrphans = 'SELECT fromTable.* ' . $sqlQuery;
        $sqlOrphansCount = 'SELECT COUNT(*) ' . $sqlQuery;

        // At the first loop, compute the total count of orphan references
        if ($total_count < 0) {
            $total_count = (int) JprestaUtils::dbGetValue($sqlOrphansCount);
        }

        $stmt = $this->pdo->query($sqlOrphans);

        if (!$stmt) {
            JprestaUtils::addLog('Doctor | Query failed: ' . $sqlOrphans, 3);

            return 0;
        }

        try {
            $currentIndex = 0;
            while ($row = $stmt->fetch(\PDO::FETCH_ASSOC)) {
                // Skip until reaching the resume index
                if ($currentIndex++ < $index) {
                    continue;
                }

                // Check if the process should stop
                if ($this->mustStop()) {
                    $done = $index;
                    $percent = round(100 * $done / $total_count, 1, PHP_ROUND_HALF_DOWN);
                    throw new TimeoutException($module->l('Execution of control in progress: {percent}% ({done} / {total})', 'dbrelationcontrol', ['{percent}' => $percent, '{done}' => $done, '{total}' => $total_count]), ['index' => $currentIndex, 'total_count' => $total_count]);
                }

                // Missing reference value
                $missingRef = isset($row[$this->config->dbRelation->fromField])
                    ? $row[$this->config->dbRelation->fromField]
                    : null;

                // Collect PK values
                $pkValues = [];
                foreach ($this->config->dbRelation->fromPkColumns as $pkColumn) {
                    $pkValues[$pkColumn] = isset($row[$pkColumn]) ? $row[$pkColumn] : null;
                }

                // Build human-readable primary key string
                $primaryKeys = [];
                foreach ($this->config->dbRelation->fromPkColumns as $pkColumn) {
                    $primaryKeys[] = sprintf('%s=%s', $pkColumn, $this->valToStr(isset($row[$pkColumn]) ? $row[$pkColumn] : null));
                }
                $primaryKey = sprintf('(%s)', implode(', ', $primaryKeys));

                // Problem description
                $desc = sprintf(
                    $module->l('The row %s references the value "%s" which does not exist in %s.%s', 'dbrelationcontrol'),
                    $primaryKey,
                    $this->valToStr($missingRef),
                    $this->config->dbRelation->toTable,
                    $this->config->dbRelation->toField
                );

                // Create Problem
                $problem = new Problem();
                $problem->problem_class = 'DbRelationProblem';
                $problem->problem_description = $desc;
                $problem->problem_config = json_encode([
                    'config' => $this->config,
                    'pkValues' => $pkValues,
                ]);
                $problem->state = !empty($this->config->canFix)
                    ? Problem::STATE_TO_FIX
                    : Problem::STATE_CANNOT_FIX;

                // Save the Problem
                call_user_func($onProblem, $problem);
            }
        } finally {
            $stmt->closeCursor();
        }

        return $currentIndex;
    }

    // -----------------------------------------------------------------
    // Helpers
    // -----------------------------------------------------------------

    /**
     * Format a value for human-readable output (not SQL).
     *
     * @param mixed $v Value to format
     * @param int $maxLength Optional max length for display truncation
     *
     * @return string
     */
    private function valToStr($v, $maxLength = 0)
    {
        if ($v === null) {
            return 'NULL';
        }
        if ($v === '') {
            return "''";
        }

        $s = (string) $v;

        return ($maxLength > 0) ? substr($s, 0, $maxLength) : $s;
    }

    /**
     * Escape a SQL identifier (column/table) with backticks, doubling inner backticks.
     *
     * Example: my`col → `my``col`
     *
     * @param string $identifier
     *
     * @return string
     */
    private static function qid($identifier)
    {
        return '`' . str_replace('`', '``', $identifier) . '`';
    }

    /**
     * Escape a table name with prefix and backticks.
     * If $table already starts with $prefix, the prefix is not duplicated.
     *
     * Examples:
     *  tableWithPrefixEscaped('customer', 'ps_')    → `ps_customer`
     *  tableWithPrefixEscaped('ps_customer', 'ps_') → `ps_customer`
     *  tableWithPrefixEscaped('orders', '')         → `orders`
     *
     * @param string $table Table name without prefix
     * @param string $prefix PrestaShop prefix
     *
     * @return string
     */
    private static function tableWithPrefixEscaped($table, $prefix)
    {
        $needsPrefix = ($prefix !== '' && strncmp($table, $prefix, strlen($prefix)) !== 0);
        $real = $needsPrefix ? ($prefix . $table) : $table;

        return self::qid($real);
    }
}
