<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed Pack are powered by Jpresta (jpresta.com)
 *
 * @author    Jpresta
 * @copyright Jpresta
 * @license   See the license of this module in file LICENSE.txt
 */

namespace JPresta\Doctor\Control;

if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * Represents a database relation between two columns.
 *
 * This class describes the mapping between a source table/column (the "from" side)
 * and a target table/column (the "to" side). It also provides metadata about
 * constraints, defaults, and conditions used when validating or repairing relations.
 */
class DbRelation
{
    // -----------------------------------------------------------------
    // Core relation definition
    // -----------------------------------------------------------------

    /**
     * @var string name of the source table (the one containing the foreign key)
     */
    public $fromTable;

    /**
     * @var string name of the source field (foreign key column)
     */
    public $fromField;

    /**
     * @var string name of the target table (the one referenced)
     */
    public $toTable;

    /**
     * @var string name of the target field (primary key or unique column)
     */
    public $toField;

    // -----------------------------------------------------------------
    // Source table metadata
    // -----------------------------------------------------------------

    /**
     * @var string[] list of primary key columns of the source table
     */
    public $fromPkColumns = [];

    /**
     * @var string[] up to four string-like columns of the source table
     *               (used for display/debug purposes)
     */
    public $fromStringColumns = [];

    /**
     * @var bool whether the source field is required (NOT NULL in database schema)
     */
    public $fromFieldIsRequired = false;

    /**
     * @var bool whether the value `0` should be considered as NULL (undefined)
     */
    public $fromFieldZeroIsNull = true;

    /**
     * @var mixed default value of the source field (raw value from database schema)
     */
    public $fromFieldDefaultValue;

    // -----------------------------------------------------------------
    // Optional conditions
    // -----------------------------------------------------------------

    /**
     * @var string|null Name of a column typically called `deleted` or `is_deleted`.
     *                  If provided, rows where this column = 1 will be ignored
     *                  when checking relations, in order to skip "soft-deleted" data.
     */
    public $deletedColumn;

    /**
     * Build a DbRelation instance from an associative array.
     *
     * @param array $data the array of the relation values
     *
     * @return DbRelation
     */
    public static function buildFromArray(array $data)
    {
        $relation = new DbRelation();
        foreach ($data as $key => $value) {
            if (property_exists($relation, $key)) {
                $relation->$key = $value;
            }
        }

        return $relation;
    }
}
