<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed Pack are powered by Jpresta (jpresta.com)
 *
 * @author    Jpresta
 * @copyright Jpresta
 * @license   See the license of this module in file LICENSE.txt
 */

namespace JPresta\Doctor\Command;

if (!defined('_PS_VERSION_')) {
    exit;
}

use JPresta\Doctor\Domain\Problem;
use JPresta\Doctor\Exception\TimeoutException;
use JPresta\Doctor\Service\DoctorService;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

class CronFixProblemsCommand extends Command
{
    protected function configure()
    {
        $this->setName('jprestadoctor:cron-fix-problems')
            ->setDescription('Fix problems that are enabled for CRON jobs');
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $moduleName = basename(dirname(__DIR__, 2));
        if ($moduleName === 'jprestadoctorp') {
            $output->writeln('| You need to upgrade to JPresta Doctor PRO version to use this command.');
        } else {
            $output->writeln('| Fixing problems for CRON jobs...');

            $moduleInstance = \Module::getInstanceByName($moduleName);
            $problemsIds = Problem::getAllIdsForCron();
            if (!$problemsIds) {
                $output->writeln('| No problems to fix');
            } else {
                $output->writeln('| Found ' . count($problemsIds) . ' problems to fix');
            }
            foreach ($problemsIds as $problemId) {
                try {
                    $startTime = microtime(true);

                    $currentProblem = new Problem($problemId);
                    if (!\Validate::isLoadedObject($currentProblem)) {
                        throw new \PrestaShopException("id_problem=$problemId not found");
                    }
                    $output->writeln("| Running \"$currentProblem->problem_description\"...");

                    $messages = [];
                    $finished = false;
                    while (!$finished) {
                        try {
                            $messages = DoctorService::fixProblemById($moduleInstance, $problemId, 60);
                            $finished = true;
                        } catch (TimeoutException $e) {
                            $output->writeln('| [in_progress]: ' . $e->getMessage());
                        }
                    }
                    $runDurationMs = round((microtime(true) - $startTime) * 1000, 0, PHP_ROUND_HALF_DOWN);
                    $output->writeln("| => done in $runDurationMs ms");
                    foreach ($messages as $messageType => $messagesStr) {
                        foreach ($messagesStr as $messageStr) {
                            $output->writeln("| [$messageType]: $messageStr");
                        }
                    }
                    $output->writeln('|_____________');
                } catch (\Exception $e) {
                    $output->writeln('| *** Error: ' . $e->getMessage());
                }
            }
            $output->writeln('| Done.');
        }

        // Return OK
        return 0;
    }
}
