<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
 *
 *    @author    Jpresta
 *    @copyright Jpresta
 *    @license   See the license of this module in file LICENSE.txt, thank you.
 */

use JPresta\SpeedPack\JprestaUtils;

if (!defined('_PS_VERSION_')) {
    exit;
}

// If the user installs jprestadoctorpro over jprestadoctor, we need to avoid error so we do not
// load the classes again. The installation of jprestadoctorpro will uninstall jprestadoctor
if (!class_exists('JPresta\Doctor\JprestaDoctorModule')) {
    require_once __DIR__ . '/classes/JprestaUtils.php';
    require_once __DIR__ . '/classes/JprestaDoctorModule.php';
    require_once __DIR__ . '/src/Domain/Control.php';
    require_once __DIR__ . '/src/Domain/Problem.php';
    require_once __DIR__ . '/src/Domain/AbstractControl.php';
    require_once __DIR__ . '/src/Domain/AbstractProblem.php';
    require_once __DIR__ . '/src/Control/DbRelation.php';
    require_once __DIR__ . '/src/Control/DbRelationControl.php';
    require_once __DIR__ . '/src/Control/DbRelationControlConfig.php';
    require_once __DIR__ . '/src/Control/OptimizeTablesControl.php';
    require_once __DIR__ . '/src/Control/PurgeAbandonedCartsControl.php';
    require_once __DIR__ . '/src/Control/PurgeOldGuestsControl.php';
    require_once __DIR__ . '/src/Control/PurgeConnectionLogsControl.php';
    require_once __DIR__ . '/src/Control/PurgeLogsControl.php';
    require_once __DIR__ . '/src/Control/PurgePageNotFoundControl.php';
}

class Jprestadoctor extends Module
{
    public $jpresta_submodules = [];

    public function __construct()
    {
        $this->name = 'jprestadoctor';
        $this->tab = 'administration';
        $this->version = '1.1.1';
        $this->author = 'Jpresta.com';
        $this->ps_versions_compliancy = ['min' => '1.6', 'max' => '9.999.999'];
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = 'JPresta - Doctor';
        $this->description = $this->l('Run a set of checks to diagnose and repair your Prestashop.');

        // Create submodules instances
        $this->jpresta_submodules['JprestaDoctorModule'] = new JPresta\Doctor\JprestaDoctorModule($this);
    }

    public function getContent()
    {
        $link = $this->context->link->getAdminLink('AdminJprestaDoctorControls');
        if (Tools::version_compare(_PS_VERSION_, '1.7', '>')) {
            Tools::redirect($link);
        } else {
            // There is a bug in redirect and getAdminLink in PS1.5 and PS1.6 so we do it ourselves
            $path = parse_url($_SERVER['REQUEST_URI'])['path'];
            header('Location: //' . $_SERVER['HTTP_HOST'] . dirname($path) . '/' . $link);
            exit;
        }
    }

    public function install()
    {
        // Check PS version compliancy first
        if (method_exists($this, 'checkCompliancy') && !$this->checkCompliancy()) {
            $this->_errors[] = $this->l('The version of your module is not compliant with your PrestaShop version.');

            return false;
        }

        if (!$this->checkIntegratedModules()) {
            return false;
        }

        if (!$this->checkIncompatibleModules()) {
            return false;
        }

        // Install module
        $install_ok = parent::install();
        if ($install_ok) {
            try {
                // Install submodules
                $this->jpresta_submodules['JprestaDoctorModule']->install();

                if (JprestaUtils::version_compare(_PS_VERSION_, '1.6', '>')) {
                    $idTab = (int) Tab::getIdFromClassName('AdminAdvancedParameters');
                    if (!$idTab) {
                        $idTab = (int) Tab::getIdFromClassName('AdminTools');
                    }
                    $this->installTab('AdminJprestaDoctorControls', [
                        'es' => 'JPresta - Doctor' . ($this->name != 'jprestadoctor' ? ' PRO' : ''),
                        'it' => 'JPresta - Dottore' . ($this->name != 'jprestadoctor' ? ' PRO' : ''),
                        'pl' => 'JPresta - Doktor' . ($this->name != 'jprestadoctor' ? ' PRO' : ''),
                        'de' => 'JPresta - Doktor' . ($this->name != 'jprestadoctor' ? ' PRO' : ''),
                        'fr' => 'JPresta - Docteur' . ($this->name != 'jprestadoctor' ? ' PRO' : ''),
                        'en' => 'JPresta - Doctor' . ($this->name != 'jprestadoctor' ? ' PRO' : ''),
                    ], $idTab);
                } elseif (JprestaUtils::version_compare(_PS_VERSION_, '1.5', '>')) {
                    $this->installTab('AdminJprestaDoctorControls', [
                        'es' => 'JPresta - Doctor' . ($this->name != 'jprestadoctor' ? ' PRO' : ''),
                        'it' => 'JPresta - Dottore' . ($this->name != 'jprestadoctor' ? ' PRO' : ''),
                        'pl' => 'JPresta - Doktor' . ($this->name != 'jprestadoctor' ? ' PRO' : ''),
                        'de' => 'JPresta - Doktor' . ($this->name != 'jprestadoctor' ? ' PRO' : ''),
                        'fr' => 'JPresta - Docteur' . ($this->name != 'jprestadoctor' ? ' PRO' : ''),
                        'en' => 'JPresta - Doctor' . ($this->name != 'jprestadoctor' ? ' PRO' : ''),
                    ], 17);
                }
            } catch (PrestaShopException $e) {
                $install_ok = false;
                $this->_errors[] = $e->getMessage() . '. ' . $this->l('Please, contact the support of this module with this error message.');
                try {
                    // An error occured while setting up the module, uninstall it to avoid a bad installation
                    parent::uninstall();
                } catch (PrestaShopException $e2) {
                    JprestaUtils::addLog('Cannot uninstall module ' . $this->name . ' after having this error during installation: "' . $e->getMessage() . '"" -> Got this error: ' . $e2->getMessage(), 4);
                }
            }
        }

        return (bool) $install_ok;
    }

    protected function checkIntegratedModules()
    {
        if ($this->name === 'jprestadoctor') {
            return true;
        }

        $moduleDir = _PS_MODULE_DIR_;
        $current = $this->name;

        // true = silently uninstall if found
        // false = block installation
        $integratedModules = [
            'jprestadoctor' => true,
        ];

        foreach ($integratedModules as $modName => $silentUninstall) {
            if ($modName === $current) {
                continue;
            }

            $modPath = $moduleDir . $modName;
            $isInstalled = Module::isInstalled($modName);
            $existsOnDisk = file_exists($modPath);

            if ($isInstalled || $existsOnDisk) {
                // Clean orphaned entries
                if (!$existsOnDisk) {
                    JprestaUtils::dbExecuteSQL('DELETE FROM `' . _DB_PREFIX_ . 'module` WHERE `name` = \'' . pSQL($modName) . '\'');
                    continue;
                }

                $instance = Module::getInstanceByName($modName);
                $display = $instance ? $instance->displayName : $modName;

                if ($silentUninstall) {
                    JprestaUtils::addLog('Uninstalling integrated module "' . $display . '"');
                    if ($instance) {
                        $instance->uninstall();
                    }
                    JprestaUtils::deleteDirectory($modPath);
                } else {
                    $this->_errors[] = sprintf(
                        $this->l('Before installing this module you must uninstall "%s" module and delete its directory: %s', 'jprestadoctor'),
                        $display,
                        $modPath
                    );

                    return false;
                }
            }
        }

        return true;
    }

    protected function checkIncompatibleModules()
    {
        $moduleDir = _PS_MODULE_DIR_;
        $incompatibleModules = [
            'jprestaspeedpack',
        ];
        if ($this->name == 'jprestadoctor') {
            $incompatibleModules[] = 'jprestadoctorpro';
        }
        $detected = [];

        foreach ($incompatibleModules as $modName) {
            if (Module::isInstalled($modName) && file_exists($moduleDir . $modName)) {
                $instance = Module::getInstanceByName($modName);
                $display = $instance ? $instance->displayName : $modName;
                $detected[] = $display . ' (' . $modName . ')';
            }
        }

        if (!empty($detected)) {
            $msg = $this->l('The following modules are incompatible with this one. Please uninstall them before continuing:', 'jprestadoctor') .
                "\n - " . implode("\n - ", $detected);
            $this->_errors[] = nl2br($msg);

            return false;
        }

        return true;
    }

    public function installTab($adminController, $name = false, $id_parent = -1)
    {
        $isUpdate = true;
        $tab = Tab::getInstanceFromClassName($adminController);
        if (!$tab || !$tab->id) {
            $tab = new Tab();
            $tab->class_name = $adminController;
            $isUpdate = false;
        }
        $tab->active = 1;
        $tab->name = [];
        foreach (Language::getLanguages(true) as $lang) {
            // Translation for modules are cached in a global variable but the local is ignored >:(
            if (is_array($name)) {
                if (array_key_exists($lang['iso_code'], $name)) {
                    $trans = $name[$lang['iso_code']];
                } elseif (array_key_exists('en', $name)) {
                    $trans = $name['en'];
                } else {
                    $trans = false;
                }
            } else {
                $trans = $name;
            }
            $tab->name[$lang['id_lang']] = !$trans ? $this->name : $trans;
        }
        $tab->id_parent = $id_parent;
        $tab->module = $this->name;
        if ($isUpdate) {
            return $tab->update();
        } else {
            return $tab->add();
        }
    }

    public function uninstallTab($adminController)
    {
        $id_tab = (int) Tab::getIdFromClassName($adminController);
        if ($id_tab) {
            $tab = new Tab($id_tab);
            if (Validate::isLoadedObject($tab)) {
                return $tab->delete();
            } else {
                $return = false;
            }
        } else {
            $return = true;
        }

        return $return;
    }

    private function uninstallAllTab()
    {
        $tabs = Tab::getCollectionFromModule($this->name);
        if (JprestaUtils::isIterable($tabs)) {
            foreach ($tabs as $tab) {
                $tab->delete();
            }
        }

        return true;
    }

    public function checkTabAccesses($adminController)
    {
        try {
            if (Tools::version_compare(_PS_VERSION_, '1.7', '>=')) {
                $slug = Access::sluggifyTab(['class_name' => $adminController], 'READ');
                $granted = Access::isGranted($slug, $this->context->employee->id_profile);
                if (!$granted) {
                    $id_role = JprestaUtils::dbGetValue('SELECT `id_authorization_role` FROM `' . _DB_PREFIX_ . 'authorization_role` WHERE slug = \'' . pSql($slug) . '\'');
                    if ($id_role) {
                        $sql = '
                        INSERT IGNORE INTO `' . _DB_PREFIX_ . 'access` (`id_profile`, `id_authorization_role`)
                        VALUES (' . (int) $this->context->employee->id_profile . ',' . (int) $id_role . ')
                    ';
                        Db::getInstance()->execute($sql);
                    }
                }
            } else {
                $id_tab = Tab::getIdFromClassName($adminController);
                $profile = Profile::getProfileAccess($this->context->employee->id_profile, $id_tab);
                if (!$profile['view']) {
                    $sql = 'UPDATE `' . _DB_PREFIX_ . 'access` SET `view`=1, `add`=1, `edit`=1, `delete`=1 WHERE id_profile=' . (int) $this->context->employee->id_profile . ' AND id_tab=' . (int) $id_tab;
                    Db::getInstance()->execute($sql);
                }
            }
        } catch (Exception $e) {
            // ignore
            JprestaUtils::addLog('Doctor | Error in checkTabAccesses(): ' . $e->getMessage(), 1);
        }
    }

    public function needsUpgrade()
    {
        if (Tools::version_compare(_PS_VERSION_, '1.7', '>')) {
            $database_version = JprestaUtils::dbGetValue('SELECT version FROM `' . _DB_PREFIX_ . 'module` WHERE name=\'' . pSQL($this->name) . '\'');

            return Tools::version_compare($this->version, $database_version, '>');
        }

        return false;
    }

    public function upgradeIfNeeded()
    {
        if ($this->needsUpgrade()) {
            $moduleManagerBuilder = PrestaShop\PrestaShop\Core\Addon\Module\ModuleManagerBuilder::getInstance();
            $moduleManager = $moduleManagerBuilder->build();

            // Clearing the cache is really long so I disable it. If needed the user will do it manually.
            $moduleManager->setActionParams(['cacheClearEnabled' => false]);

            try {
                // Upgrade the module
                if ($moduleManager->upgrade($this->name)) {
                    $this->_confirmations[] = $this->l('The module has been upgraded to version', 'jprestadoctor') . ' ' . $this->version;
                } else {
                    $this->_errors[] = $this->l('The module could not be upgraded correctly, contact the support with above error messages', 'jprestadoctor');
                }
            } catch (Exception $e) {
                // Just ignore
            }
        }
    }

    /**
     * Override Module::updateModuleTranslations()
     */
    public function updateModuleTranslations()
    {
        // Speeds up installation: do nothing because this module translation are not in Prestashop language pack
    }

    public function disable($force_all = false)
    {
        $ret = parent::disable($force_all);
        // disable submodules
        $ret &= $this->jpresta_submodules['JprestaDoctorModule']->disable();

        return (bool) $ret;
    }

    public function enable($force_all = false)
    {
        $ret = parent::enable($force_all);
        // enable submodules
        $ret &= $this->jpresta_submodules['JprestaDoctorModule']->enable();

        return (bool) $ret;
    }

    public function uninstall()
    {
        $ret = parent::uninstall();
        // Uninstall submodules
        $this->jpresta_submodules['JprestaDoctorModule']->uninstall();

        return (bool) $ret;
    }

    public function l($string, $specific = false, $locale = null)
    {
        if ($specific === 'Admin.Global') {
            $parameters = [];
            $parameters['legacy'] = 'htmlspecialchars';

            return $this->getTranslator()->trans($string, $parameters, $specific, null);
        } else {
            if (is_array($locale)) {
                // If locale is an array then consider it as parameters for sprintf
                return Translate::getModuleTranslation(
                    $this,
                    $string,
                    ($specific) ? $specific : $this->name,
                    $locale,
                    false,
                    null
                );
            }

            return parent::l($string, $specific, $locale);
        }
    }
}
