<?php
/**
 * JPresta Doctor, JPresta Doctor PRO and Speed pack are powered by Jpresta (jpresta . com)
 *
 *    @author    Jpresta
 *    @copyright Jpresta
 *    @license   See the license of this module in file LICENSE.txt, thank you.
 */

require_once __DIR__ . '/../../src/Domain/AbstractProblem.php';
require_once __DIR__ . '/../../src/Domain/Problem.php';
require_once __DIR__ . '/../../src/Service/DoctorService.php';
require_once __DIR__ . '/../../classes/JprestaUtils.php';

// Automatically include Control and Problem classes so get_declared_classes() will return them
foreach (['Control', 'Problem'] as $dir) {
    $path = __DIR__ . '/../../src/' . $dir . '/*.php';

    foreach (glob($path) as $filename) {
        if (basename($filename) !== 'index.php') {
            require_once $filename;
        }
    }
}

use JPresta\Doctor\Domain\Control;
use JPresta\Doctor\Domain\Problem;
use JPresta\Doctor\Exception\TimeoutException;
use JPresta\Doctor\Service\DoctorService;
use JPresta\SpeedPack\JprestaUtils;

if (!defined('_PS_VERSION_')) {
    exit;
}

class AdminJprestaDoctorControlsController extends ModuleAdminController
{
    public function __construct()
    {
        parent::__construct();
        $this->bootstrap = true;
        $this->context = Context::getContext();
    }

    /**
     * Check if the module needs to be upgraded (scripts)
     */
    public function init()
    {
        if ($this->module->needsUpgrade()) {
            $this->module->upgradeIfNeeded();
            $this->confirmations = array_merge($this->confirmations, $this->module->getConfirmations());
            $this->errors = array_merge($this->errors, $this->module->getErrors());
        }
        parent::init();
    }

    public function ajaxProcessEvaluate()
    {
        header('Content-Type: application/json; charset=utf-8');
        $infos = [];
        try {
            $currentControl = null;
            $nextControl = null;
            $messages = [];
            $problemCount = 0;
            $runDurationMs = 0;
            if (!Tools::getIsset('id_control')) {
                // Full diagnostic, get the first task
                $id_control = Control::getFirst();
                if ($id_control) {
                    $nextControl = new Control($id_control);
                }
            } else {
                $id_control = (int) Tools::getValue('id_control');
                $startTime = microtime(true);
                try {
                    $problemCount += DoctorService::runControlById($this->module, $id_control, 60);
                    $currentControl = new Control($id_control);
                    $next_id_control = Control::getNext($id_control);
                    if ($next_id_control) {
                        $nextControl = new Control($next_id_control);
                    }
                } catch (TimeoutException $e) {
                    $messages['info'][] = $e->getMessage();
                    $currentControl = $nextControl = new Control($id_control);
                }
                $runDurationMs = (microtime(true) - $startTime) * 1000;
            }
            $infos['total'] = Control::getCountActive();
            $problemCounts = [];
            $problemCounts[Problem::STATE_TO_FIX] = Problem::getCountByState(Problem::STATE_TO_FIX);
            $problemCounts[Problem::STATE_CANNOT_FIX] = Problem::getCountByState(Problem::STATE_CANNOT_FIX);
            $problemCounts[Problem::STATE_FIXED] = Problem::getCountByState(Problem::STATE_FIXED);
            $problemCounts[Problem::STATE_ERROR_IN_FIX] = Problem::getCountByState(Problem::STATE_ERROR_IN_FIX);
            $infos['problem_counts'] = $problemCounts;
            $infos['last_full_check'] = Control::getLastFullCheckAsString();
            if ($currentControl != null) {
                $infos['done'] = [];
                $infos['done']['id_control'] = $currentControl->id_control;
                $infos['done']['name'] = $currentControl->control_name;
                $infos['done']['problem_count'] = $problemCount;
                $infos['done']['duration_ms'] = round($runDurationMs, 0, PHP_ROUND_HALF_DOWN);
                $infos['done']['messages'] = $messages;
            }
            if ($nextControl != null) {
                $infos['next'] = [];
                $infos['next']['id_control'] = $nextControl->id_control;
                $infos['next']['name'] = $nextControl->control_name;
            }
        } catch (Exception $e) {
            $infos['error'] = $e->getMessage();
            $infos['error_trace'] = JprestaUtils::jTraceEx($e);
            http_response_code(500);
        } catch (Error $e) {
            $infos['error'] = $e->getMessage();
            $infos['error_trace'] = JprestaUtils::jTraceEx($e);
            http_response_code(500);
        }
        exit(json_encode($infos));
    }

    public function ajaxProcessDeleteControl()
    {
        header('Content-Type: application/json; charset=utf-8');
        $infos = [];
        try {
            $id_control = (int) Tools::getValue('id_control');
            $control = new Control($id_control);
            if (!Validate::isLoadedObject($control)) {
                throw new PrestaShopException("id_control=$id_control not found");
            }
            $control->delete();
        } catch (Exception $e) {
            $infos['error'] = $e->getMessage();
            $infos['error_trace'] = JprestaUtils::jTraceEx($e);
            http_response_code(500);
        } catch (Error $e) {
            $infos['error'] = $e->getMessage();
            $infos['error_trace'] = JprestaUtils::jTraceEx($e);
            http_response_code(500);
        }
        exit(json_encode($infos));
    }

    public function ajaxProcessDeleteProblem()
    {
        header('Content-Type: application/json; charset=utf-8');
        $infos = [];
        try {
            $id_problem = (int) Tools::getValue('id_problem');
            $problem = new Problem($id_problem);
            if (!Validate::isLoadedObject($problem)) {
                throw new PrestaShopException("id_problem=$id_problem not found");
            }
            $problem->delete();
        } catch (Exception $e) {
            $infos['error'] = $e->getMessage();
            $infos['error_trace'] = JprestaUtils::jTraceEx($e);
            http_response_code(500);
        } catch (Error $e) {
            $infos['error'] = $e->getMessage();
            $infos['error_trace'] = JprestaUtils::jTraceEx($e);
            http_response_code(500);
        }
        exit(json_encode($infos));
    }

    public function ajaxProcessRepair()
    {
        header('Content-Type: application/json; charset=utf-8');
        $infos = [];
        try {
            $currentProblem = null;
            $nextProblem = null;
            $messages = [];
            $runDurationMs = 0;
            $id_problem = Tools::getValue('id_problem', null);
            $id_control = Tools::getValue('id_control', null);
            if (!$id_problem) {
                $id_problem = Problem::getFirst($id_control);
                if ($id_problem) {
                    $nextProblem = new Problem($id_problem);
                }
            } else {
                $id_problem = (int) Tools::getValue('id_problem');
                $startTime = microtime(true);
                try {
                    $messages = DoctorService::fixProblemById($this->module, $id_problem, 60);
                    $currentProblem = new Problem($id_problem);
                    $next_id_problem = Problem::getNext($id_problem);
                    if ($next_id_problem) {
                        $nextProblem = new Problem($next_id_problem);
                        if ($id_control && $nextProblem->id_control != $id_control) {
                            // id_control was defined, but the next problem is not in the same control
                            // so we stop here
                            $nextProblem = null;
                        }
                    }
                } catch (TimeoutException $e) {
                    $messages['info'][] = $e->getMessage();
                    $currentProblem = $nextProblem = new Problem($id_problem);
                }
                $runDurationMs = (microtime(true) - $startTime) * 1000;
            }
            $infos['total'] = Problem::getCountByState(Problem::STATE_TO_FIX, $id_control);
            $problemCounts = [];
            $problemCounts[Problem::STATE_TO_FIX] = Problem::getCountByState(Problem::STATE_TO_FIX);
            $problemCounts[Problem::STATE_CANNOT_FIX] = Problem::getCountByState(Problem::STATE_CANNOT_FIX);
            $problemCounts[Problem::STATE_FIXED] = Problem::getCountByState(Problem::STATE_FIXED);
            $problemCounts[Problem::STATE_ERROR_IN_FIX] = Problem::getCountByState(Problem::STATE_ERROR_IN_FIX);
            $infos['problem_counts'] = $problemCounts;
            $infos['last_full_check'] = Control::getLastFullCheckAsString();
            if ($currentProblem != null) {
                $infos['done'] = [];
                $infos['done']['id_problem'] = $currentProblem->id_problem;
                $infos['done']['name'] = $currentProblem->problem_description;
                $infos['done']['duration_ms'] = round($runDurationMs, 0, PHP_ROUND_HALF_DOWN);
                $infos['done']['state'] = $currentProblem->state;
                $infos['done']['messages'] = $messages;
            }
            if ($nextProblem != null) {
                $infos['next'] = [];
                $infos['next']['id_problem'] = $nextProblem->id_problem;
                $infos['next']['id_control'] = $id_control;
                $infos['next']['name'] = $nextProblem->problem_description;
            }
        } catch (Exception $e) {
            $infos['error'] = $e->getMessage();
            $infos['error_trace'] = JprestaUtils::jTraceEx($e);
            http_response_code(500);
        } catch (Error $e) {
            $infos['error'] = $e->getMessage();
            $infos['error_trace'] = JprestaUtils::jTraceEx($e);
            http_response_code(500);
        }
        exit(json_encode($infos));
    }

    public function ajaxProcessGetStats()
    {
        header('Content-Type: application/json; charset=utf-8');
        $infos = [];
        try {
            $infos['control_count'] = Control::getCountActive();
            $infos['problem_count'] = Problem::getCountNotFixed();
            $problemCounts = [];
            $problemCounts[Problem::STATE_TO_FIX] = Problem::getCountByState(Problem::STATE_TO_FIX);
            $problemCounts[Problem::STATE_CANNOT_FIX] = Problem::getCountByState(Problem::STATE_CANNOT_FIX);
            $problemCounts[Problem::STATE_FIXED] = Problem::getCountByState(Problem::STATE_FIXED);
            $problemCounts[Problem::STATE_ERROR_IN_FIX] = Problem::getCountByState(Problem::STATE_ERROR_IN_FIX);
            $infos['problem_counts'] = $problemCounts;
            $infos['last_full_check'] = Control::getLastFullCheckAsString();
        } catch (Exception $e) {
            $infos['error'] = $e->getMessage();
            $infos['error_trace'] = JprestaUtils::jTraceEx($e);
            http_response_code(500);
        } catch (Error $e) {
            $infos['error'] = $e->getMessage();
            $infos['error_trace'] = JprestaUtils::jTraceEx($e);
            http_response_code(500);
        }
        exit(json_encode($infos));
    }

    public function ajaxProcessResetControls()
    {
        header('Content-Type: application/json; charset=utf-8');
        $infos = [];
        try {
            $infos = DoctorService::resetControls($this->module);
        } catch (Exception $e) {
            $infos['error'] = $e->getMessage();
            $infos['error_trace'] = JprestaUtils::jTraceEx($e);
            http_response_code(500);
        } catch (Error $e) {
            $infos['error'] = $e->getMessage();
            $infos['error_trace'] = JprestaUtils::jTraceEx($e);
            http_response_code(500);
        }
        exit(json_encode($infos));
    }

    public function setMedia($isNewTheme = false)
    {
        parent::setMedia($isNewTheme);
        if ($this->module->name === 'jprestaspeedpack') {
            $this->addCSS(__PS_BASE_URI__ . 'modules/' . $this->module->name . '/views/css/back.css?v=' . $this->module->version, 'all', null, false);
        }
        $this->addCSS(__PS_BASE_URI__ . 'modules/' . $this->module->name . '/views/css/jquery.dataTables.min.css?v=' . $this->module->version, 'all', null, false);
        $this->addCSS(__PS_BASE_URI__ . 'modules/' . $this->module->name . '/views/css/buttons.dataTables.min.css?v=' . $this->module->version, 'all', null, false);
        $this->addCSS(__PS_BASE_URI__ . 'modules/' . $this->module->name . '/views/css/doctor.css?v=' . $this->module->version, 'all', null, false);
        $this->addCSS(__PS_BASE_URI__ . 'modules/' . $this->module->name . '/views/css/doctor-flipcounter.css?v=' . $this->module->version, 'all', null, false);
        $this->addJS(__PS_BASE_URI__ . 'modules/' . $this->module->name . '/views/js/jquery.dataTables.min.js?v=' . $this->module->version, false);
        $this->addJS(__PS_BASE_URI__ . 'modules/' . $this->module->name . '/views/js/dataTables.buttons.min.js?v=' . $this->module->version, false);
        $this->addJS(__PS_BASE_URI__ . 'modules/' . $this->module->name . '/views/js/doctor.js?v=' . $this->module->version, false);
        $this->addJS(__PS_BASE_URI__ . 'modules/' . $this->module->name . '/views/js/doctor-flipcounter.js?v=' . $this->module->version, false);
    }

    private function needsSettingsReview()
    {
        foreach (get_declared_classes() as $class) {
            if (is_subclass_of($class, JPresta\Doctor\Control\AbstractControl::class)) {
                if (method_exists($class, 'getFormFields')) {
                    try {
                        $fields = $class::getFormFields($this->module);
                        foreach ($fields as $field) {
                            if (!JprestaUtils::getConfigurationAllShop($field['name'] . '_reviewed')) {
                                return true;
                            }
                        }
                    } catch (Exception $e) {
                        JprestaUtils::addLog($e->getMessage(), 2, 'jprestadoctor');
                    }
                }
            }
        }
    }

    public function generateSettingsForm()
    {
        $helper = new HelperForm();
        $helper->module = $this->module;

        $fieldsForm = [];
        $allFields = [];
        foreach (get_declared_classes() as $class) {
            if (is_subclass_of($class, JPresta\Doctor\Control\AbstractControl::class)) {
                if (method_exists($class, 'getFormFields')) {
                    try {
                        $fields = $class::getFormFields($this->module);
                        if (count($fields) > 0) {
                            foreach ($fields as &$field) {
                                $helper->fields_value[$field['name']] = JprestaUtils::getConfigurationAllShop($field['name']);
                                if (!JprestaUtils::getConfigurationAllShop($field['name'] . '_reviewed')) {
                                    if (isset($field['class'])) {
                                        $field['class'] .= ' need_review';
                                    } else {
                                        $field['class'] = 'need_review';
                                    }
                                }
                            }
                            $allFields = array_merge($allFields, $fields);
                        }
                    } catch (Exception $e) {
                        JprestaUtils::addLog($e->getMessage(), 2, 'jprestadoctor');
                    }
                }
            }
        }
        if (count($allFields) === 0) {
            return '';
        }

        $fieldsForm[0]['form'] = [
            'legend' => [
                'title' => $this->module->l('Settings', 'Admin.Global'),
                'icon' => 'icon-cogs',
            ],
            'input' => $allFields,
            'buttons' => [
                ['title' => $this->module->l('Save', 'Admin.Actions'),
                    'type' => 'submit',
                    'icon' => 'process-icon-save',
                    'class' => 'btn-primary pull-right',
                    'name' => 'submitDoctor'],
            ],
        ];

        return $helper->generateForm($fieldsForm);
    }

    public function postProcess()
    {
        if (Tools::isSubmit('submitDoctor')) {
            foreach (get_declared_classes() as $class) {
                if (is_subclass_of($class, JPresta\Doctor\Control\AbstractControl::class)) {
                    if (method_exists($class, 'getFormFields')) {
                        try {
                            $fields = $class::getFormFields($this->module);
                            foreach ($fields as $field) {
                                JprestaUtils::saveConfigurationAllShop($field['name'], Tools::getValue($field['name']));
                                JprestaUtils::saveConfigurationAllShop($field['name'] . '_reviewed', 1);
                            }
                        } catch (Exception $e) {
                            JprestaUtils::addLog($e->getMessage(), 2, 'jprestadoctor');
                        }
                    }
                }
            }
        }
        parent::postProcess();
    }

    public function renderList()
    {
        $this->module->checkTabAccesses('AdminJprestaDoctorControlsDatas');
        $this->module->checkTabAccesses('AdminJprestaDoctorProblemsDatas');

        // Add JS def here so it is after postProcess
        $problemCounts = [];
        $problemCounts[Problem::STATE_TO_FIX] = Problem::getCountByState(Problem::STATE_TO_FIX);
        $problemCounts[Problem::STATE_CANNOT_FIX] = Problem::getCountByState(Problem::STATE_CANNOT_FIX);
        $problemCounts[Problem::STATE_FIXED] = Problem::getCountByState(Problem::STATE_FIXED);
        $problemCounts[Problem::STATE_ERROR_IN_FIX] = Problem::getCountByState(Problem::STATE_ERROR_IN_FIX);
        Media::addJsDef([
            'doctor_ajax_url' => $this->context->link->getAdminLink('AdminJprestaDoctorControls'),
            'doctor_datas_controls_url' => $this->context->link->getAdminLink('AdminJprestaDoctorControlsDatas'),
            'doctor_datas_problems_url' => $this->context->link->getAdminLink('AdminJprestaDoctorProblemsDatas'),
            'last_full_check' => Control::getLastFullCheckAsString(),
            'problem_count' => Problem::getCountNotFixed(),
            'problem_counts' => $problemCounts,
            'control_count' => Control::getCountActive(),
            'can_repair' => $this->module->name === 'jprestadoctorpro' || $this->module->name === 'jprestaspeedpack' || file_exists(_PS_MODULE_DIR_ . $this->module->name . '/src/Problem/DbRelationProblem.php'),
            'needs_settings_review' => $this->needsSettingsReview(),
        ]);

        $infos = [];
        $infos['module_name'] = $this->module->name;
        $infos['faq_url'] = 'https://jpresta.com/' . Context::getContext()->language->iso_code . '/faq/doctor';
        $infos['modulePath'] = __PS_BASE_URI__ . 'modules/' . $this->module->name;
        $infos['problem_count'] = Problem::getCountNotFixed();
        $infos['control_count'] = Control::getCountActive();
        $infos['settings_form'] = $this->generateSettingsForm();
        $infos['needs_settings_review'] = $this->needsSettingsReview();
        $infos['cron_urls_legacy'] = $this->getCronURLsLegacy();

        $this->context->smarty->assign($infos);

        return $this->context->smarty->fetch(_PS_MODULE_DIR_ . $this->module->name . '/views/templates/admin/doctor-controls.tpl');
    }

    private function getCronURLsLegacy()
    {
        $urls = [];
        if ($this->module->name === 'jprestaspeedpack') {
            foreach (Shop::getContextListShopID() as $shopId) {
                $shop = new Shop($shopId);
                $url = $shop->getBaseURL(true);
                if (JprestaUtils::strlen($url) > 0) {
                    $urls[] = $url . '?fc=module&module=' . $this->module->name . '&controller=dboptimize&token=' . JprestaUtils::getSecurityToken($shopId);
                }
            }
        }

        return $urls;
    }
}
